package rollbar

import (
	"bytes"
	"fmt"
)

type itemToSend struct {
	AccessToken string   `json:"access_token"`
	Data        sendData `json:"data"`
}

type sendData struct {
	Environment string `json:"environment"`
	*Data
}

type Data struct {
	Body *Body `json:"body"`
	DataOptionals
}

type DataOptionals struct {
	Level       Level             `json:"level,omitempty"`
	Timestamp   int64             `json:"timestamp,omitempty"`
	CodeVersion string            `json:"code_version,omitempty"`
	Platform    string            `json:"platform,omitempty"`
	Language    string            `json:"language,omitempty"`
	Server      Server            `json:"server,omitempty"`
	Fingerprint string            `json:"fingerprint,omitempty"`
	Title       string            `json:"title,omitempty"`
	Notifier    Notifier          `json:"notifier,omitempty"`
	Custom      map[string]string `json:"custom,omitempty"`
}

type DeploymentOptions struct {
	LocalUsername   *string
	RollbarUsername *string
	Comment         *string
	Environment     *string
}

// Server signals the sending server.  See rollbar for documentation.
type Server struct {
	Host string `json:"host"`
}

// Notifier signals the sending library.  See rollbar for documentation.
type Notifier struct {
	Name    string `json:"string"`
	Version string `json:"version"`
}

// Level is the log level sent
type Level string

const (
	// Critical log level
	Critical = Level("critical")
	// Error log level
	Error = Level("error")
	// Warning log level
	Warning = Level("warning")
	// Info log level
	Info = Level("info")
	// Debug log level
	Debug = Level("debug")
)

// Frame is a stack trace
type Frame struct {
	Filename string `json:"filename"`
	Method   string `json:"method"`
	Line     int    `json:"lineno"`
}

// Exception is the error sent.
type Exception struct {
	Class   string `json:"class"`
	Message string `json:"message"`
}

// Trace is used to send errors to rollbar
type Trace struct {
	Frames    []Frame   `json:"frames"`
	Exception Exception `json:"exception"`
}

// Message is a normal, non error log message
type Message struct {
	Body string `json:"body"`
}

// Body is the configurable part sent to rollbar.
type Body struct {
	Message *Message `json:"message,omitempty"`
	Trace   *Trace   `json:"trace,omitempty"`
}

type deploymentResponse struct {
	Data map[string]string `json:"data,omitempty"`
}

// MergeFrom will merge the parameter into this object
func (d *DataOptionals) MergeFrom(from *DataOptionals) {
	if from == nil {
		d.MergeFrom(&DefaultConfiguration)
		return
	}
	if d.CodeVersion == "" {
		d.CodeVersion = from.CodeVersion
	}
	if d.Platform == "" {
		d.Platform = from.Platform
	}
	if d.Language == "" {
		d.Language = from.Language
	}
	if d.Server.Host == "" {
		d.Server.Host = from.Server.Host
	}
	if d.Notifier.Name == "" {
		d.Notifier.Name = from.Notifier.Name
	}
	if d.Notifier.Version == "" {
		d.Notifier.Version = from.Notifier.Version
	}
	d.mergeCustom(from)
}

func (d *DataOptionals) mergeCustom(from *DataOptionals) {
	if d.Custom == nil {
		d.Custom = from.Custom
		return
	}
	for k, v := range from.Custom {
		if _, exists := d.Custom[k]; !exists {
			d.Custom[k] = v
		}
	}
}

type deploymentPostBody struct {
	LocalUsername   *string `json:"local_username,omitempty"`
	RollbarUsername *string `json:"rollbar_username,omitempty"`
	Comment         *string `json:"comment,omitempty"`
	Environment     string  `json:"environment"`
	Revision        string  `json:"revision"`
	AccessToken     string  `json:"access_token"`
}

type responseErr struct {
	body         bytes.Buffer
	wrapped      error
	httpcode     int
	responseCode int
}

func (r *responseErr) Error() string {
	return fmt.Sprintf("rollbar client error: httpcode[%d] responseCode[%d] body[%s] wrapped[%s]", r.httpcode, r.responseCode, r.body.String(), r.wrap())
}

func (r *responseErr) wrap() string {
	if r.wrapped == nil {
		return ""
	}
	return r.wrapped.Error()
}
