#!/bin/bash
set -ex

function json_read() {
  python -c '
import sys, json
j=json.load(sys.stdin)
if "'$1'" in j: print j["'$1'"]
'
}

if [ -z "$COMMIT_TO_DEPLOY" ]; then
  if [ ! -z "$GIT_COMMIT" ]; then
    COMMIT_TO_DEPLOY="$GIT_COMMIT"
  elif [ -e .scratch/deploy_version ]; then
    COMMIT_TO_DEPLOY=$(cat .scratch/deploy_version)
  fi
fi

if [ -z "$TEAM" ]; then
  (>&2 echo "Team name not set")
  exit 1
fi

ASSUME_ROLE="$(json_read deploy_aws_role < deploy/$TEAM/environments/$ENVIRONMENT/deploy.json)"
PROMOTE_FROM="$(json_read promote_from < deploy/$TEAM/environments/$ENVIRONMENT/deploy.json)"
AWS_CREDS="$(json_read aws_creds < deploy/$TEAM/environments/$ENVIRONMENT/deploy.json)"
IMAGE_PATH="docker-registry.internal.justin.tv/$TEAM/$PROJECT"
BUILD_PIPELINE_VERSION="98c5da424b6fcf4edaf1ded07f9ca676d46e4d6b"
if [ -e .scratch/terraform.json ]; then
  REGION="$(json_read region < .scratch/terraform.json)"
  CLUSTER="$(json_read cluster_name < .scratch/terraform.json)"
  SERVICE="$(json_read service_name < .scratch/terraform.json)"
fi

if [ -e .scratch/task_definition ]; then
  TASK=$(cat .scratch/task_definition)
fi

function do_jenkins() {
  echo "Starting jenkins build"
  env
  rm -rf .scratch
  mkdir .scratch

  if [ ! -z "$AWS_CREDS" ]; then
    echo "$AWS_CREDS" > next_creds
  fi
  echo "read_terraform" > next_step
  echo "read terraform information" > next_desc

}

function do_validate_docker_image() {
  echo "Validating the docker image"
  docker pull "$IMAGE_PATH:$COMMIT_TO_DEPLOY"

  echo "create_task" > next_step
  echo "create ECS task" > next_desc
}

function docker_run() {
  ADD_CONFIG_ENV=""
  if [ ! -z "$AWS_CONFIG_FILE" ]; then
    ADD_CONFIG_ENV="-e AWS_CONFIG_FILE=/awsconfig -v $AWS_CONFIG_FILE:/awsconfig:ro"
  fi
  echo 'docker run --read-only -e AWS_SDK_LOAD_CONFIG=1 '$ADD_CONFIG_ENV' -v '$(pwd)':/twitch:ro docker-registry.internal.justin.tv'
}

function do_read_terraform() {
  TERRAFORM="terraform/$TEAM/environments/$ENVIRONMENT/$PROJECT/main.tf"
  $(docker_run)/deploy/tformstates3:$BUILD_PIPELINE_VERSION /tformstates3 -terraform /twitch/$TERRAFORM -assume_role "$ASSUME_ROLE" > .scratch/terraform.json
  cat .scratch/terraform.json

  TERRAFORM="terraform/$TEAM/environments/$ENVIRONMENT/core/main.tf"
  $(docker_run)/deploy/tformstates3:$BUILD_PIPELINE_VERSION /tformstates3 -terraform /twitch/$TERRAFORM -assume_role "$ASSUME_ROLE" > .scratch/core_terraform.json
  cat .scratch/core_terraform.json
  echo "validate_docker_image" > next_step
  echo "Validating docker image to deploy" > next_desc

  if [ -z "$COMMIT_TO_DEPLOY" ]; then
    if [ ! -z "$GIT_COMMIT" ]; then
      COMMIT_TO_DEPLOY="$GIT_COMMIT"
    elif [ ! -z "$PROMOTE_FROM" ]; then
      echo "Detecting commit to promote from $PROMOTE_FROM"
      docker run -e CONSUL_HTTP_ADDR=consul.internal.justin.tv consul kv get -datacenter us-west2 "pipeline/deployed/$TEAM/$PROJECT/$PROMOTE_FROM" > .scratch/deploy_version
      echo "Detected deploy version $(cat .scratch/deploy_version)"
    else
      (>&2 echo "Could not detect a version to deploy")
      exit 1
    fi
  fi
}

function do_create_task() {
  TEMPLATE="deploy/$TEAM/$PROJECT/task_template.json"
  if [ ! -e "$TEMPLATE" ]; then
    TEMPLATE="deploy/common/task_template.json"
  fi
  echo "{\"Region\": \"$REGION\", \"git_commit\":\"$COMMIT_TO_DEPLOY\", \"Environment\":\"$ENVIRONMENT\", \"Image\": \"$IMAGE_PATH:$COMMIT_TO_DEPLOY\"}" > .scratch/build_params.json
  $(docker_run)/deploy/createtask:$BUILD_PIPELINE_VERSION /createtask -template /twitch/$TEMPLATE -region $REGION -verbose -params /twitch/.scratch/build_params.json:/twitch/.scratch/terraform.json:/twitch/.scratch/core_terraform.json -assume_role "$ASSUME_ROLE" -clean > .scratch/task_definition
  cat .scratch/task_definition
  echo "deploy_ecs_task" > next_step
  echo "Starting deploy" > next_desc
}

function do_deploy_ecs_task() {
  echo "here is where we deploy"
  $(docker_run)/deploy/deploytask:$BUILD_PIPELINE_VERSION /deploytask -region $REGION -cluster $CLUSTER -service $SERVICE -task $TASK -assume_role "$ASSUME_ROLE" -verbose
  echo "wait_till_stable" > next_step
  echo "Waiting for deployed task to stabilize" > next_desc
}

function do_wait_till_stable() {
  echo "here is where we wait till the task is stable"
  $(docker_run)/deploy/waituntilstable:$BUILD_PIPELINE_VERSION /waituntilstable -region $REGION -cluster $CLUSTER -service $SERVICE -assume_role "$ASSUME_ROLE" -verbose
  echo "post_deploy_tests" > next_step
  echo "Run post deploy tests" > next_desc
}

function do_post_deploy_tests() {
  echo "Here is where I would run integration tests, auto check graphs, etc"
  echo "tag_image" > next_step
  echo "Tagging deployed version" > next_desc
}

function do_tag_image() {
  echo "Tagging image as deployed"
  docker pull "$IMAGE_PATH:$COMMIT_TO_DEPLOY"
  docker tag "$IMAGE_PATH:$COMMIT_TO_DEPLOY" "$IMAGE_PATH:$ENVIRONMENT"
  docker push "$IMAGE_PATH:$ENVIRONMENT"
  docker run -e CONSUL_HTTP_ADDR=consul.internal.justin.tv consul kv put -datacenter us-west2 "pipeline/deployed/$TEAM/$PROJECT/$ENVIRONMENT" $COMMIT_TO_DEPLOY
  echo "done" > next_step
}

function do_all() {
  echo "This should run the entire pipeline starting at jenkins..."
  exit 1
}

case "$1" in
  jenkins)
    do_jenkins
    ;;
  validate_docker_image)
    do_validate_docker_image
    ;;
  read_terraform)
    do_read_terraform
    ;;
  create_task)
    do_create_task
    ;;
  deploy_ecs_task)
    do_deploy_ecs_task
    ;;
  wait_till_stable)
    do_wait_till_stable
    ;;
  tag_image)
    do_tag_image
    ;;
  post_deploy_tests)
    do_post_deploy_tests
    ;;
  all)
    do_all
    ;;
  *)
  echo "Usage: $0 {...}"
    exit 1
    ;;
esac
