package main

import (
	"bytes"
	"encoding/json"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"strings"
	"text/template"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/log/fmtlogger"
	"code.justin.tv/twitch/gocode/src/deploy/taskmgmt"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awsutil"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ecs"
	"github.com/aws/aws-sdk-go/service/sts"
)

type deployer struct {
	flagSet     *flag.FlagSet
	commandArgs []string
	errOut      io.Writer
	out         io.Writer
	osExit      func(int)
	verboseLog  log.Logger
}

var instance = deployer{
	flagSet:     flag.NewFlagSet(os.Args[0], flag.ExitOnError),
	commandArgs: os.Args[1:],
	errOut:      os.Stderr,
	osExit:      os.Exit,
	out:         os.Stdout,
}

func main() {
	instance.flagSet.Usage = func() {
		fmt.Println("Creates a new task definition from a Go template file")
		instance.flagSet.PrintDefaults()
	}
	instance.main()
}

type parsedParams struct {
	taskTemplate string
	region       string
	params       string
	awsRole      string
	verbose      bool
	dryrun       bool
	cleanTasks   bool
}

func (d *deployer) main() {
	if err := d.run(); err != nil {
		fmt.Fprintln(d.errOut, err.Error())
		d.osExit(1)
	}
}

func (d *deployer) createTaskDefinitionInput(parsedParamsInst *parsedParams) (*ecs.RegisterTaskDefinitionInput, error) {
	templateParams := map[string]interface{}{}
	templateParams["Region"] = parsedParamsInst.region
	for _, filename := range strings.Split(parsedParamsInst.params, ":") {
		filename = strings.TrimSpace(filename)
		if filename == "" {
			continue
		}
		paramsBytes, err := ioutil.ReadFile(filename)
		if err != nil {
			return nil, errors.Wrapf(err, "unable to read containers file %s", filename)
		}
		buf := bytes.Buffer{}
		buf.Write(paramsBytes)
		theseParams := make(map[string]interface{})
		if err := json.NewDecoder(&buf).Decode(&theseParams); err != nil {
			return nil, errors.Wrapf(err, "Unable to parse template parameters file %s", filename)
		}
		for k, v := range theseParams {
			templateParams[k] = v
		}
	}

	b, _ := json.Marshal(templateParams)
	d.verboseLog.Log("template_params", string(b))
	containerTemplate, err := ioutil.ReadFile(parsedParamsInst.taskTemplate)
	if err != nil {
		return nil, errors.Wrapf(err, "Unable to read containers file %s", parsedParamsInst.taskTemplate)
	}

	tmplate, err := template.New("containers").Parse(string(containerTemplate))
	if err != nil {
		return nil, errors.Wrapf(err, "Unable to parse containers template %s", parsedParamsInst.taskTemplate)
	}
	buf := bytes.Buffer{}
	if err := tmplate.Execute(&buf, templateParams); err != nil {
		return nil, errors.Wrap(err, "Unable to process template")
	}

	d.verboseLog.Log("template", buf.String(), "template parsed")

	abc := ecs.RegisterTaskDefinitionInput{}
	if err := json.NewDecoder(&buf).Decode(&abc); err != nil {
		return nil, errors.Wrap(err, "Unable to read container template file as containers")
	}
	return &abc, nil
}

func (d *deployer) getTaskFamily(abc *ecs.RegisterTaskDefinitionInput, dryrun bool, verbose bool, ecsClient *ecs.ECS) (string, error) {
	if dryrun {
		return *abc.Family, nil
	}
	req, out2 := ecsClient.RegisterTaskDefinitionRequest(abc)
	if err2 := req.Send(); err2 != nil {
		return "", errors.Wrap(err2, "Unable to register task definition")
	}
	if verbose {
		if err2 := json.NewEncoder(d.errOut).Encode(out2.TaskDefinition); err2 != nil {
			return "", errors.Wrap(err2, "Unable to output task definition result")
		}
	}
	fmt.Fprintf(d.out, "%s:%d\n", *out2.TaskDefinition.Family, *out2.TaskDefinition.Revision)
	return *out2.TaskDefinition.Family, nil
}

func (d *deployer) run() error {
	parsedParamsInst := parsedParams{}
	d.flagSet.StringVar(&parsedParamsInst.region, "region", "", "aws region to run commands in")
	d.flagSet.BoolVar(&parsedParamsInst.verbose, "verbose", false, "verbose output")
	d.flagSet.BoolVar(&parsedParamsInst.cleanTasks, "clean", false, "clean old tasks after creation")
	d.flagSet.BoolVar(&parsedParamsInst.dryrun, "dryrun", false, "Just print what it would do, without doing it")
	d.flagSet.StringVar(&parsedParamsInst.taskTemplate, "template", "", "task definition template file")
	d.flagSet.StringVar(&parsedParamsInst.params, "params", "", ": separated file(s) of JSON to use as parameters to task creation")
	d.flagSet.StringVar(&parsedParamsInst.awsRole, "assume_role", "", "An AWS role to assume")
	err := d.flagSet.Parse(d.commandArgs)
	if err != nil {
		return errors.Wrap(err, "Unable to parse command line arguments")
	}

	errLog := fmtlogger.NewLogfmtLogger(d.errOut, log.Discard)
	if parsedParamsInst.verbose {
		d.verboseLog = fmtlogger.NewLogfmtLogger(d.errOut, log.Discard)
	} else {
		d.verboseLog = log.Discard
	}

	abc, err := d.createTaskDefinitionInput(&parsedParamsInst)
	if err != nil {
		return err
	}

	d.verboseLog.Log("container", abc, "container definitions loaded")

	configs := []*aws.Config{}
	configs = append(configs, &aws.Config{
		Region: &parsedParamsInst.region,
	})
	if parsedParamsInst.awsRole != "" {
		awsSessionForRole, err2 := session.NewSession(configs...)
		if err != nil {
			return errors.Wrap(err2, "Unable to create aws session for assumed role")
		}
		stsclient := sts.New(awsSessionForRole)
		arp := &stscreds.AssumeRoleProvider{
			ExpiryWindow: 10 * time.Second,
			RoleARN:      parsedParamsInst.awsRole,
			Client:       stsclient,
		}
		credentials := credentials.NewCredentials(arp)
		configs = append(configs, &aws.Config{
			Credentials: credentials,
		})
	}

	awsSession, err := session.NewSession(configs...)

	if err != nil {
		return errors.Wrap(err, "Unable to create aws session")
	}
	ecsClient := ecs.New(awsSession)
	taskFamily, err := d.getTaskFamily(abc, parsedParamsInst.dryrun, parsedParamsInst.verbose, ecsClient)
	if err != nil {
		return err
	}

	if parsedParamsInst.cleanTasks {
		cleaner := taskmgmt.CleanupTasks{
			ECSClient:   ecsClient,
			VerboseLog:  d.verboseLog,
			ErrLog:      errLog,
			TasksToKeep: 20,
			Dryrun:      parsedParamsInst.dryrun,
		}
		removed, err := cleaner.Run(taskFamily)
		if err != nil {
			fmt.Fprintf(d.errOut, "Unable to clean task definitions: %s\n", err)
		} else {
			d.verboseLog.Log("removed", awsutil.Prettify(removed), "items removed")
		}
	}
	return nil
}
