package main

import (
	"net"
	"os"
	"os/signal"
	"time"

	"code.justin.tv/feeds/service-common"
	"code.justin.tv/twitch/gocode/src/deploy/cmd/deployer/internal/httprpc"
	"code.justin.tv/twitch/gocode/src/deploy/cmd/deployer/internal/httprpc/apiv1"
)

const (
	teamName    = "deploy"
	serviceName = "deployer"
)

// CodeVersion is set by build script
var CodeVersion string

type service struct {
	osExit        func(code int)
	onListen      func(net.Addr)
	sigChan       chan os.Signal
	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner

	server httprpc.Server

	configs struct {
		HTTPConfig httprpc.Config
	}
}

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion: CodeVersion,
	},
}

func (f *service) setup() error {
	if err := f.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(f.serviceCommon.Config, &f.configs.HTTPConfig); err != nil {
		return err
	}

	return nil
}

func (f *service) inject() {
	f.server = httprpc.Server{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &f.configs.HTTPConfig.BaseHTTPServerConfig,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
			Dims:        &f.serviceCommon.CtxDimensions,
			Log:         f.serviceCommon.Log,
			ElevateKey:  f.serviceCommon.ElevateLogKey,
			Ctxlog:      &f.serviceCommon.Ctxlog,
			OnListen:    f.onListen,
			PanicLogger: f.serviceCommon.PanicLogger,
			XRay:        f.serviceCommon.XRay,
		},
		APIv1: apiv1.API{},
	}
	f.server.SetupRoutes = f.server.DeployerRoutes
	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.server, &f.serviceCommon,
		},
		SigChan:      f.sigChan,
		SignalNotify: signal.Notify,
	}
	// Cannot just pass f because f contains private members that I cannot nil check via reflection
	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"aws-sdk-go", "net/http"},
	}).Check(f, &f.server, f.configs, f.runner)
	res.MustBeEmpty(os.Stderr)
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		if f.serviceCommon.Log != nil {
			f.serviceCommon.Log.Log("err", err, "Unable to load initial config")
		}
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		time.Sleep(time.Millisecond * 150)
		f.osExit(1)
		return
	}
	f.inject()
	if err := f.runner.Execute(); err != nil {
		f.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		time.Sleep(time.Millisecond * 150)
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished main")
}

func main() {
	instance.main()
}
