package main

import (
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"os"
	"os/user"
	"path/filepath"
	"sort"
	"strings"
	"sync"
	"text/template"
	"time"

	"code.justin.tv/feeds/errors"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ecs"
	"github.com/aws/aws-sdk-go/service/iam"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/fatih/color"
	"github.com/hashicorp/consul/api"
	"github.com/hashicorp/terraform/terraform"
	"github.com/urfave/cli"
	"golang.org/x/sync/errgroup"
)

type dataAPI struct {
	coreConfigs []*aws.Config
	sessions    map[string]*cachedSession

	onceKv sync.Once
	kv     *api.KV
	kvErr  error
}

type cachedSession struct {
	err     error
	session *session.Session
}

type terraformStateRemote struct {
	Region string `json:"region"`
	Bucket string `json:"bucket"`
	Key    string `json:"key"`
}

type deploymentInfo struct {
	DeployAWSRole  string               `json:"deploy_aws_role"`
	AWSCreds       string               `json:"aws_creds"`
	Region         string               `json:"region"`
	PromoteFrom    string               `json:"promote_from"`
	Profile        string               `json:"profile"`
	TerraformState terraformStateRemote `json:"terraform_state"`
}

func (d *dataAPI) getDeploymentInfo(team string, service string, env string) (*deploymentInfo, error) {
	kv, err := d.createConsulClient()
	if err != nil {
		return nil, err
	}
	p, _, err := kv.Get("pipeline/info/"+team+"/"+env, nil)
	if err != nil {
		return nil, err
	}
	if p == nil {
		return nil, errors.New("unable to find consul key")
	}
	var ret deploymentInfo
	if err := json.Unmarshal(p.Value, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

func (d *dataAPI) getIAMClient(region string, awsRole string, profile string) (*iam.IAM, error) {
	awsSession, err := d.getAWSSession(region, "", profile)
	if err != nil {
		return nil, err
	}
	iamClient := iam.New(awsSession)
	return iamClient, nil
}

func (d *dataAPI) getECSClient(region string, awsRole string, profile string) (*ecs.ECS, error) {
	awsSession, err := d.getAWSSession(region, awsRole, profile)
	if err != nil {
		return nil, err
	}
	ecsClient := ecs.New(awsSession)
	return ecsClient, nil
}

func (d *dataAPI) getS3Client(region string, awsRole string, profile string) (*s3.S3, error) {
	awsSession, err := d.getAWSSession(region, awsRole, profile)
	if err != nil {
		return nil, err
	}
	s3client := s3.New(awsSession)
	return s3client, nil
}

func (d *dataAPI) getAWSSession(region string, awsRole string, profile string) (*session.Session, error) {
	if d.sessions == nil {
		d.sessions = map[string]*cachedSession{}
	}
	key := region + "_" + awsRole + "_" + profile
	if prev, exists := d.sessions[key]; exists {
		return prev.session, prev.err
	}
	theseConfigs := append(make([]*aws.Config, 0, len(d.coreConfigs)+1), d.coreConfigs...)
	theseConfigs = append(theseConfigs, &aws.Config{
		Region: &region,
	})
	stored := cachedSession{}
	d.sessions[key] = &stored
	return stored.create(region, awsRole, profile, theseConfigs)
}

func (c *cachedSession) create(region string, awsRole string, profile string, coreConfigs []*aws.Config) (*session.Session, error) {
	if c.session == nil && c.err == nil {
		c.session, c.err = func() (*session.Session, error) {
			opts := session.Options{
				SharedConfigState: session.SharedConfigEnable,
				Profile:           profile,
			}
			opts.Config.MergeIn(coreConfigs...)
			awsSessionForRole, err := session.NewSessionWithOptions(opts)
			if err != nil {
				return nil, err
			}
			if awsRole == "" {
				return awsSessionForRole, nil
			}
			stsclient := sts.New(awsSessionForRole)
			arp := &stscreds.AssumeRoleProvider{
				ExpiryWindow: 10 * time.Second,
				RoleARN:      awsRole,
				Client:       stsclient,
			}
			credentials := credentials.NewCredentials(arp)
			theseConfigs := append(make([]*aws.Config, 0, len(coreConfigs)+1), coreConfigs...)
			theseConfigs = append(theseConfigs, &aws.Config{
				Credentials: credentials,
			})
			return session.NewSession(theseConfigs...)
		}()
	}
	return c.session, c.err
}

func (d *dataAPI) createConsulClient() (*api.KV, error) {
	d.onceKv.Do(func() {
		config := api.DefaultConfig()
		config.Address = "http://consul.internal.justin.tv"
		config.Datacenter = "us-west2"

		client, err := api.NewClient(config)
		if err != nil {
			d.kvErr = err
			return
		}
		d.kv = client.KV()
	})
	return d.kv, d.kvErr
}

func (d *dataAPI) lsDir(dir string) ([]string, error) {
	kv, err := d.createConsulClient()
	if err != nil {
		return nil, err
	}
	pairs, _, err := kv.Keys("/"+dir+"/", "/", nil)
	if err != nil {
		return nil, err
	}
	return trimmeKeys([]string{dir + "/", dir}, pairs), nil
}

func trimmeKeys(prefix []string, keys []string) []string {
	ret := make([]string, 0, len(keys))
	for _, k := range keys {
		k = strings.TrimSuffix(k, "/")
		added := false
		for _, p := range prefix {
			trimmed := strings.TrimPrefix(k, p)
			if trimmed != k {
				if trimmed != "" {
					ret = append(ret, trimmed)
				}
				added = true
				break
			}
		}
		if !added {
			ret = append(ret, k)
		}
	}
	return ret
}

func (d *dataAPI) teams() ([]string, error) {
	return d.lsDir("pipeline/deployed")
}

func (d *dataAPI) teamServices(team string) ([]string, error) {
	return d.lsDir("pipeline/deployed/" + team)
}

func (d *dataAPI) deployedVersion(team string, service string, env string) (string, error) {
	kv, err := d.createConsulClient()
	if err != nil {
		return "", err
	}
	pair, _, err := kv.Get("pipeline/deployed/"+team+"/"+service+"/"+env, nil)
	if err != nil {
		return "", err
	}
	if pair == nil {
		return "", errors.New("unable to find consul key")
	}
	return string(pair.Value), nil
}

func (d *dataAPI) serviceEnvs(team string, service string) ([]string, error) {
	return sortEnvs(d.lsDir("pipeline/deployed/" + team + "/" + service + "/"))
}

var envToIndex = map[string]int{
	"latest":      1,
	"integration": 2,
	"staging":     3,
	"canary":      4,
	"production":  5,
}

func sortEnvs(envs []string, err error) ([]string, error) {
	if err != nil {
		return envs, err
	}
	sort.Slice(envs, func(i, j int) bool {
		i1 := envToIndex[envs[i]]
		i2 := envToIndex[envs[j]]
		return i1 < i2
	})
	return envs, err
}

func (d *dataAPI) services() ([]string, error) {
	teams, err := d.teams()
	if err != nil {
		return nil, err
	}
	var mu sync.Mutex
	s := make([]string, 0, len(teams)*10)
	eg := errgroup.Group{}
	for _, team := range teams {
		team := team
		eg.Go(func() error {
			svcs, err := d.teamServices(team)
			if err != nil {
				return err
			}
			mu.Lock()
			for _, svc := range svcs {
				s = append(s, team+"-"+svc)
			}
			mu.Unlock()
			return nil
		})
	}
	return s, eg.Wait()
}

type deployerApp struct {
	app      *cli.App
	data     dataAPI
	verbose  bool
	skiprole bool
}

var instance = deployerApp{
	app: cli.NewApp(),
}

func main() {
	instance.main()
}

func (d *deployerApp) Output(a ...interface{}) {
	fmt.Fprintln(d.app.Writer, a...)
}

func (d *deployerApp) Errorf(a ...interface{}) {
	fmt.Fprintln(d.app.ErrWriter, a...)
}

func (d *deployerApp) pipelineGetAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)

	kv, err := d.data.createConsulClient()
	if err != nil {
		return err
	}

	kvp, _, err := kv.Get(fmt.Sprintf("pipeline/override/%s/%s/%s/promote_from", team, service, env), nil)
	if kvp == nil {
		return nil
	}
	fmt.Fprintln(d.app.Writer, string(kvp.Value))

	return err
}

func (d *deployerApp) pipelineGetComplete(c *cli.Context) {
	d.teamServiceEnvComplete(c)
}

func (d *deployerApp) pipelineReleaseAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)

	kv, err := d.data.createConsulClient()
	if err != nil {
		return err
	}

	_, err = kv.Delete(fmt.Sprintf("pipeline/override/%s/%s/%s/promote_from", team, service, env), nil)

	return err
}

func (d *deployerApp) pipelineReleaseComplete(c *cli.Context) {
	d.teamServiceEnvComplete(c)
}

func (d *deployerApp) pipelineClaimAction(c *cli.Context) error {
	if c.NArg() != 4 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)
	tag := c.Args().Get(3)

	kv, err := d.data.createConsulClient()
	if err != nil {
		return err
	}

	kvpair := &api.KVPair{
		Key:   fmt.Sprintf("pipeline/override/%s/%s/%s/promote_from", team, service, env),
		Value: []byte(tag),
	}
	_, err = kv.Put(kvpair, nil)

	return err
}

func (d *deployerApp) pipelineClaimComplete(c *cli.Context) {
	d.teamServiceEnvComplete(c)
}

func (d *deployerApp) certifySetAction(c *cli.Context) error {
	if c.NArg() != 4 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)
	tag := c.Args().Get(3)

	kv, err := d.data.createConsulClient()
	if err != nil {
		return err
	}
	kvpair := &api.KVPair{
		Key:   fmt.Sprintf("pipeline/deployed/%s/%s/%s", team, service, env),
		Value: []byte(tag),
	}
	_, err = kv.Put(kvpair, nil)
	if err != nil {
		return err
	}
	fmt.Fprintln(d.app.Writer, "OK")
	return nil
}

func (d *deployerApp) certifyGetAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)
	version, err := d.data.deployedVersion(team, service, env)
	if err != nil {
		return err
	}
	fmt.Fprintln(d.app.Writer, version)
	return nil
}

func (d *deployerApp) handleCompletionError(err error) {
	d.Errorf("Invalid bash completion: %s", err.Error())
}

func (d *deployerApp) statusComplete(c *cli.Context) {
	d.teamServiceEnvComplete(c)
	fmt.Fprintln(d.app.Writer, "")
}

type statusResult struct {
	Team               string
	Service            string
	Env                string
	TerraformState     map[string]string
	ECSService         *ecs.Service
	DeployedVersion    string
	PromoteFrom        string
	PromoteFromVersion string
	Short              bool
}

func (s *statusResult) fullTemplate(out io.Writer) error {
	return template.Must(template.New("status").Parse(fullStatusTemplate)).Execute(out, s)
}

func (s *statusResult) ConsulStatus() string {
	if len(s.ECSService.Deployments) != 1 {
		return color.RedString("Being deployed")
	}
	if s.DeployedVersion != s.PromoteFromVersion {
		return color.YellowString("Undeployed")
	}

	return color.GreenString("OK")
}

func (s *statusResult) Center(text string) string {
	leftOver := 80 - len(text)
	if leftOver <= 0 {
		return text
	}
	return strings.Repeat("=", leftOver/2) + text + strings.Repeat("=", (leftOver+1)/2)
}

func (d *deployerApp) verboseLogYourself(deploymentInfo *deploymentInfo) error {
	if d.verbose {
		iamClient, err := d.data.getIAMClient(deploymentInfo.Region, "", deploymentInfo.Profile)
		if err != nil {
			return err
		}
		whoIsMakingRequest, err2 := iamClient.GetUser(&iam.GetUserInput{})
		if err2 != nil {
			return err2
		}
		fmt.Fprintln(d.app.ErrWriter, whoIsMakingRequest.User.String())
	}
	return nil
}

func (d *deployerApp) discoverDeployedService(clusterName string, serviceName string, roleToUse string, deploymentInfo *deploymentInfo) (*ecs.Service, error) {
	describeInput := &ecs.DescribeServicesInput{
		Cluster:  &clusterName,
		Services: []*string{&serviceName},
	}
	ecsClient, err := d.data.getECSClient(deploymentInfo.Region, roleToUse, deploymentInfo.Profile)
	if err != nil {
		return nil, err
	}
	servicesState, err := ecsClient.DescribeServices(describeInput)
	if err != nil {
		return nil, err
	}
	if len(servicesState.Failures) > 0 {
		return nil, errors.Errorf("service description failure: %s", *servicesState.Failures[0].Reason)
	}
	if len(servicesState.Services) == 0 {
		return nil, errors.New("unable to find service")
	}
	if len(servicesState.Services) != 1 {
		return nil, errors.New("too many services returned")
	}
	ecsService := servicesState.Services[0]
	return ecsService, nil
}

func (d *deployerApp) statusActionForEnv(team string, service string, env string, short bool) error {
	version, err := d.data.deployedVersion(team, service, env)
	if err != nil {
		return err
	}
	deploymentInfo, err := d.data.getDeploymentInfo(team, service, env)
	if err != nil {
		return err
	}
	roleToUse := deploymentInfo.AWSCreds
	if d.skiprole {
		roleToUse = ""
	}
	if err2 := d.verboseLogYourself(deploymentInfo); err2 != nil {
		return err2
	}
	s3client, err := d.data.getS3Client(deploymentInfo.TerraformState.Region, roleToUse, deploymentInfo.Profile)
	if err != nil {
		return err
	}
	currentState, err := d.readTerraformState(s3client, deploymentInfo.TerraformState.Bucket, deploymentInfo.TerraformState.Key)
	if err != nil {
		return err
	}
	clusterName := currentState["cluster_name"]
	serviceName := currentState["service_name"]

	ecsService, err := d.discoverDeployedService(clusterName, serviceName, roleToUse, deploymentInfo)
	if err != nil {
		return err
	}
	s := statusResult{
		Team:            team,
		Service:         service,
		Env:             env,
		TerraformState:  currentState,
		ECSService:      ecsService,
		DeployedVersion: version,
		PromoteFrom:     deploymentInfo.PromoteFrom,
		Short:           short,
	}
	if deploymentInfo.PromoteFrom != "" {
		fromVersion, err := d.data.deployedVersion(team, service, deploymentInfo.PromoteFrom)
		if err != nil {
			return err
		}
		s.PromoteFromVersion = fromVersion
	}
	return s.fullTemplate(d.app.Writer)
}

func (d *deployerApp) statusAction(c *cli.Context) error {
	if c.NArg() > 3 || c.NArg() < 2 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	if c.NArg() == 2 {
		envs, err := d.data.serviceEnvs(team, service)
		if err != nil {
			return cli.NewExitError(err.Error(), 1)
		}
		for _, env := range envs {
			if env == "latest" {
				continue
			}
			if err := d.statusActionForEnv(team, service, env, true); err != nil {
				return err
			}
		}
		return nil
	}
	env := c.Args().Get(2)
	return d.statusActionForEnv(team, service, env, false)
}

func (d *deployerApp) readTerraformState(s3client *s3.S3, bucket string, key string) (map[string]string, error) {
	out, err := s3client.GetObject(&s3.GetObjectInput{
		Bucket: &bucket,
		Key:    &key,
	})
	if err != nil {
		return nil, err
	}
	state := terraform.State{}
	if err := json.NewDecoder(out.Body).Decode(&state); err != nil {
		return nil, err
	}
	terraformOutputs := make(map[string]string, len(state.RootModule().Outputs))
	for k, v := range state.RootModule().Outputs {
		terraformOutputs[k], _ = v.Value.(string)
	}
	return terraformOutputs, nil
}

func (d *deployerApp) certifyGetComplete(c *cli.Context) {
	d.teamServiceEnvComplete(c)
}

func (d *deployerApp) dockerExistsComplete(c *cli.Context) {
	d.teamServiceEnvComplete(c)
}

func (d *deployerApp) dockerExistsAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	tag := c.Args().Get(2)
	dockerHost := c.GlobalString("docker")
	req, err := http.NewRequest(http.MethodGet, fmt.Sprintf("%s/v2/%s/%s/manifests/%s", dockerHost, team, service, tag), nil)
	if err != nil {
		return err
	}
	client := http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	if resp.StatusCode == http.StatusNotFound {
		fmt.Fprintln(d.app.Writer, "MISSING")
		return cli.NewExitError("", 1)
	}
	if resp.StatusCode == http.StatusOK {
		fmt.Fprintln(d.app.Writer, "OK")
		return nil
	}
	return errors.Errorf("Unexpected status code %d", resp.StatusCode)
}

func (d *deployerApp) certifySetComplete(c *cli.Context) {
	d.teamServiceEnvComplete(c)
}

func (d *deployerApp) teamServiceEnvComplete(c *cli.Context) {
	if c.NArg() == 0 {
		teams, err := d.data.teams()
		if err != nil {
			d.handleCompletionError(err)
			return
		}
		fmt.Fprintln(d.app.Writer, strings.Join(teams, "\n"))
		return
	}
	team := c.Args().Get(0)
	if c.NArg() == 1 {
		services, err := d.data.teamServices(team)
		if err != nil {
			d.handleCompletionError(err)
		}
		fmt.Fprintln(d.app.Writer, strings.Join(services, "\n"))
		return
	}
	service := c.Args().Get(1)
	if c.NArg() == 2 {
		team := c.Args().Get(0)
		envs, err := d.data.serviceEnvs(team, service)
		if err != nil {
			d.handleCompletionError(err)
		}
		fmt.Fprintln(d.app.Writer, strings.Join(envs, "\n"))
		return
	}
}

func (d *deployerApp) blockStableComplete(c *cli.Context) {
	d.teamServiceEnvComplete(c)
}

func (d *deployerApp) blockStableAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	//team := c.Args().Get(0)
	//service := c.Args().Get(1)
	//env := c.Args().Get(2)
	return nil
}

func (d *deployerApp) deploymentComplete(c *cli.Context) {
	d.teamServiceEnvComplete(c)
}

func (d *deployerApp) deploymentAction(c *cli.Context) error {
	if c.NArg() != 3 {
		return cli.ShowSubcommandHelp(c)
	}
	team := c.Args().Get(0)
	service := c.Args().Get(1)
	env := c.Args().Get(2)
	jenkinsJobName := team + "-" + service + "-" + env
	jenkinsHost := c.GlobalString("jenkins")
	buildURL := jenkinsHost + "/job/" + jenkinsJobName + "/buildWithParameters"
	usr, err := user.Current()
	if err != nil {
		return err
	}
	authFile := filepath.Join(usr.HomeDir, "/.jenkins_auth")
	fileContents, err := ioutil.ReadFile(authFile)
	if err != nil {
		return err
	}
	parts := strings.SplitN(strings.TrimSpace(string(fileContents)), ":", 2)
	username, password := parts[0], parts[1]
	req, err := http.NewRequest(http.MethodPost, buildURL, nil)
	req.SetBasicAuth(username, password)
	if err != nil {
		return err
	}
	client := http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		return err
	}
	if resp.StatusCode != http.StatusCreated {
		return errors.Errorf("unexpected jenkins job status %d: %s", resp.StatusCode, getBody(resp))
	}
	fmt.Fprintln(d.app.Writer, jenkinsHost+"/job/"+jenkinsJobName)
	return nil
}

func getBody(resp *http.Response) string {
	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return fmt.Sprintf("err: %s", err.Error())
	}
	return string(body)
}

func errExit(f func(c *cli.Context) error) func(c *cli.Context) error {
	return func(c *cli.Context) error {
		err := f(c)
		if err == nil {
			return nil
		}
		return cli.NewExitError(err.Error(), 1)
	}
}

func (d *deployerApp) main() {
	d.data.coreConfigs = []*aws.Config{
		{
			Logger: aws.LoggerFunc(func(v ...interface{}) {
				fmt.Fprintln(d.app.ErrWriter, v...)
			}),
			CredentialsChainVerboseErrors: aws.Bool(true),
		},
	}
	d.app.Name = "deployercli"
	d.app.Usage = "Deploy helper for ECS promotion pipelines"
	d.app.Version = "1.0"
	d.app.EnableBashCompletion = true
	d.app.ErrWriter = os.Stderr
	d.app.Writer = os.Stdout
	d.app.Flags = []cli.Flag{
		cli.BoolFlag{
			Name:        "verbose",
			Usage:       "Enables verbose output",
			Destination: &d.verbose,
		},
		cli.BoolFlag{
			Name:        "skiprole, r",
			Usage:       "If set, will skip elevating aws roles during deployments.",
			Destination: &d.skiprole,
		},
	}
	d.app.Commands = []cli.Command{
		{
			Name:  "certify",
			Usage: "Certify a SHA as being correctly deployed to an environment",
			Subcommands: []cli.Command{
				{
					Name:         "get",
					ArgsUsage:    "[team] [service] [environment]",
					Usage:        "Get the currently certified SHA for a deployment",
					Action:       errExit(d.certifyGetAction),
					BashComplete: d.certifyGetComplete,
				},
				{
					Name:         "set",
					ArgsUsage:    "[team] [service] [environment] [tag]",
					Usage:        "Set the currently certified SHA for a deployment",
					Action:       errExit(d.certifySetAction),
					BashComplete: d.certifySetComplete,
				},
			},
		},
		{
			Name:  "pipeline",
			Usage: "Pipeline promotion overrides",
			Subcommands: []cli.Command{
				{
					Name:         "claim",
					ArgsUsage:    "[team] [service] [environment] [tag]",
					Usage:        "Claim an environment as your own, with an overridden tag",
					Action:       errExit(d.pipelineClaimAction),
					BashComplete: d.pipelineClaimComplete,
				},
				{
					Name:         "release",
					ArgsUsage:    "[team] [service] [environment]",
					Usage:        "Release claim on an environment",
					Action:       errExit(d.pipelineReleaseAction),
					BashComplete: d.pipelineReleaseComplete,
				},
				{
					Name:         "get",
					ArgsUsage:    "[team] [service] [environment]",
					Usage:        "Get the currently overridden pipeline location",
					Action:       errExit(d.pipelineGetAction),
					BashComplete: d.pipelineGetComplete,
				},
			},
		},
		{
			Name:  "docker",
			Usage: "Docker image validation",
			Subcommands: []cli.Command{
				{
					Name:         "exists",
					ArgsUsage:    "[team] [service] [tag]",
					Usage:        "Returns exit code 0 if a docker image exists for the given tag",
					Action:       errExit(d.dockerExistsAction),
					BashComplete: d.dockerExistsComplete,
				},
			},
			Flags: []cli.Flag{
				cli.StringFlag{
					Name:  "docker",
					Usage: "Docker host",
					Value: "https://docker-registry.internal.justin.tv",
				},
			},
		},
		{
			Name:         "block-till-stable",
			Usage:        "Block till deployment is stable",
			ArgsUsage:    "[team] [service] [environment]",
			Action:       errExit(d.blockStableAction),
			BashComplete: d.blockStableComplete,
		},
		{
			Name:         "status",
			Usage:        "Show the status of a full deployment pipeline",
			ArgsUsage:    "[team] [service] (environment)",
			Action:       errExit(d.statusAction),
			BashComplete: d.statusComplete,
		},
		{
			Name:         "deploy",
			Usage:        "Trigger a deployment",
			ArgsUsage:    "[team] [service] [environment]",
			Action:       errExit(d.deploymentAction),
			BashComplete: d.deploymentComplete,
			Flags: []cli.Flag{
				cli.StringFlag{
					Name:  "jenkins",
					Usage: "Changes the jenkins host",
					Value: "https://jenkins.internal.justin.tv",
				},
				cli.StringFlag{
					Name:  "auth",
					Usage: "Auth to use for jenkins commands",
					Value: "~./jenkins_auth",
				},
			},
		},
	}
	if err := d.app.Run(os.Args); err != nil {
		os.Exit(1)
	}
}
