package main

import (
	"flag"
	"fmt"
	"io"
	"log"
	"os"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ecs"
	"github.com/aws/aws-sdk-go/service/sts"
)

type deployer struct {
	flagSet     *flag.FlagSet
	commandArgs []string
	errOut      io.Writer
	out         io.Writer
	osExit      func(int)
}

var instance = deployer{
	flagSet:     flag.NewFlagSet(os.Args[0], flag.ExitOnError),
	commandArgs: os.Args[1:],
	errOut:      os.Stderr,
	osExit:      os.Exit,
	out:         os.Stdout,
}

func main() {
	instance.flagSet.Usage = func() {
		fmt.Println("Creates a new task definition from a Go template file")
		instance.flagSet.PrintDefaults()
	}
	instance.run()
}

type parsedParams struct {
	region  string
	cluster string
	service string
	awsRole string
	name    string
	verbose bool
}

func (d *deployer) run() {
	parsedParamsInst := parsedParams{}
	d.flagSet.StringVar(&parsedParamsInst.region, "region", "", "aws region to run commands in")
	d.flagSet.StringVar(&parsedParamsInst.cluster, "cluster", "", "full ARN of cluster to update")
	d.flagSet.StringVar(&parsedParamsInst.service, "service", "", "service to update")
	d.flagSet.StringVar(&parsedParamsInst.name, "name", "", "name of container to look for promotion image from")
	d.flagSet.StringVar(&parsedParamsInst.awsRole, "assume_role", "", "An AWS role to assume")

	d.flagSet.BoolVar(&parsedParamsInst.verbose, "verbose", false, "verbose output")

	err := d.flagSet.Parse(d.commandArgs)
	if err != nil {
		fmt.Fprintf(d.errOut, "Unable to parse command line arguments: %s\n", err.Error())
		d.osExit(1)
		return
	}

	logger := log.New(d.out, "waiter", log.LstdFlags)
	configs := []*aws.Config{}
	configs = append(configs, &aws.Config{
		Region: &parsedParamsInst.region,
		Logger: aws.LoggerFunc(func(v ...interface{}) {
			logger.Println(v...)
		}),
	})
	if parsedParamsInst.awsRole != "" {
		awsSessionForRole, err2 := session.NewSession(configs...)
		if err2 != nil {
			fmt.Fprintf(d.errOut, "Unable to create aws session for assumed role: %s\n", err2.Error())
			d.osExit(1)
			return
		}
		stsclient := sts.New(awsSessionForRole)
		arp := &stscreds.AssumeRoleProvider{
			ExpiryWindow: 10 * time.Second,
			RoleARN:      parsedParamsInst.awsRole,
			Client:       stsclient,
		}
		credentials := credentials.NewCredentials(arp)
		configs = append(configs, &aws.Config{
			Credentials: credentials,
		})
	}

	awsSession, err := session.NewSession(configs...)

	if err != nil {
		fmt.Fprintf(d.errOut, "Unable to create aws session: %s\n", err.Error())
		d.osExit(1)
		return
	}
	ecsClient := ecs.New(awsSession)

	out, err := ecsClient.DescribeServices(&ecs.DescribeServicesInput{
		Cluster:  &parsedParamsInst.cluster,
		Services: []*string{&parsedParamsInst.service},
	})
	if err != nil {
		fmt.Fprintf(d.errOut, "Services never stablized: %s\n", err.Error())
		d.osExit(1)
		return
	}
	if len(out.Services) != 1 {
		fmt.Fprintln(d.errOut, "Unable to find a service!")
		d.osExit(1)
		return
	}
	tout, err := ecsClient.DescribeTaskDefinition(&ecs.DescribeTaskDefinitionInput{
		TaskDefinition: out.Services[0].TaskDefinition,
	})
	if err != nil {
		fmt.Fprintf(d.errOut, "Unable to describe task definition input: %s\n", err.Error())
		d.osExit(1)
		return
	}
	for _, container := range tout.TaskDefinition.ContainerDefinitions {
		if *container.Name == parsedParamsInst.name {
			imageParts := strings.Split(*container.Image, ":")
			fmt.Fprintf(d.out, "%s\n", imageParts[1])
			d.osExit(0)
			return
		}
	}
	fmt.Fprintln(d.errOut, "Unable to find the image you wanted to look for promotion")
	d.osExit(1)
}
