package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"io"
	"os"
	"time"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/log/fmtlogger"
	"code.justin.tv/twitch/gocode/src/deploy/taskmgmt"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ecs"
	"github.com/aws/aws-sdk-go/service/sts"
)

type deployer struct {
	flagSet     *flag.FlagSet
	commandArgs []string
	errOut      io.Writer
	out         io.Writer
	osExit      func(int)
	verboseLog  log.Logger
}

var instance = deployer{
	flagSet:     flag.NewFlagSet(os.Args[0], flag.ExitOnError),
	commandArgs: os.Args[1:],
	errOut:      os.Stderr,
	osExit:      os.Exit,
	out:         os.Stdout,
}

func main() {
	instance.flagSet.Usage = func() {
		fmt.Println("Clean up stale ecs tasks")
		instance.flagSet.PrintDefaults()
	}
	instance.main()
}

type parsedParams struct {
	task        string
	region      string
	awsRole     string
	tasksToKeep int
	verbose     bool
	dryrun      bool
}

func (d *deployer) main() {
	if err := d.run(); err != nil {
		fmt.Fprintln(d.errOut, err.Error())
		d.osExit(1)
	}
}

func (d *deployer) run() error {
	parsedParamsInst := parsedParams{}
	d.flagSet.StringVar(&parsedParamsInst.region, "region", "", "aws region to run commands in")
	d.flagSet.StringVar(&parsedParamsInst.task, "task", "", "task definition to cleanup")
	d.flagSet.BoolVar(&parsedParamsInst.dryrun, "dryrun", false, "Just print out what it would remove, without removing")
	d.flagSet.BoolVar(&parsedParamsInst.verbose, "verbose", false, "Verbose output")
	d.flagSet.StringVar(&parsedParamsInst.awsRole, "assume_role", "", "An AWS role to assume")
	d.flagSet.IntVar(&parsedParamsInst.tasksToKeep, "tasks_to_keep", 20, "Number of recent ECS tasks to keep")
	err := d.flagSet.Parse(d.commandArgs)
	if err != nil {
		return errors.Wrap(err, "Unable to parse command line arguments")
	}
	if parsedParamsInst.task == "" {
		return errors.New("task parameter required")
	}

	errLog := fmtlogger.NewLogfmtLogger(d.errOut, log.Discard)
	if parsedParamsInst.verbose {
		d.verboseLog = fmtlogger.NewLogfmtLogger(d.errOut, log.Discard)
	} else {
		d.verboseLog = log.Discard
	}

	configs := []*aws.Config{}
	configs = append(configs, &aws.Config{
		Region: &parsedParamsInst.region,
	})
	if parsedParamsInst.awsRole != "" {
		awsSessionForRole, err2 := session.NewSession(configs...)
		if err2 != nil {
			return errors.Wrap(err2, "unable to create aws session for assumed role")
		}
		stsclient := sts.New(awsSessionForRole)
		arp := &stscreds.AssumeRoleProvider{
			ExpiryWindow: 10 * time.Second,
			RoleARN:      parsedParamsInst.awsRole,
			Client:       stsclient,
		}
		credentials := credentials.NewCredentials(arp)
		configs = append(configs, &aws.Config{
			Credentials: credentials,
		})
	}

	awsSession, err := session.NewSession(configs...)

	if err != nil {
		return errors.Wrap(err, "unable to create aws session")
	}

	ecsClient := ecs.New(awsSession)

	cleaner := taskmgmt.CleanupTasks{
		ECSClient:   ecsClient,
		VerboseLog:  d.verboseLog,
		ErrLog:      errLog,
		TasksToKeep: 20,
		Dryrun:      parsedParamsInst.dryrun,
	}
	removedTasks, err := cleaner.Run(parsedParamsInst.task)
	if err != nil {
		return err
	}

	if err := json.NewEncoder(d.out).Encode(removedTasks); err != nil {
		fmt.Fprintf(d.errOut, "unable to JSON out which tasks are removed: %s\n", err.Error())
		d.osExit(1)
	}
	return nil
}
