package main

import (
	"flag"
	"fmt"
	"io"
	"os"

	"os/user"
	"path/filepath"
	"text/template"

	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/log/fmtlogger"
)

type deployer struct {
	flagSet     *flag.FlagSet
	commandArgs []string
	errOut      io.Writer
	out         io.Writer
	osExit      func(int)
	verboseLog  log.Logger
	errLog      log.Logger
}

var instance = deployer{
	flagSet:     flag.NewFlagSet(os.Args[0], flag.ExitOnError),
	commandArgs: os.Args[1:],
	errOut:      os.Stderr,
	osExit:      os.Exit,
	out:         os.Stdout,
}

func main() {
	instance.flagSet.Usage = func() {
		fmt.Println("Clean up stale ecs tasks")
		instance.flagSet.PrintDefaults()
	}
	instance.main()
}

type teamEnvData map[string]environmentData

type environmentData struct {
	Region  string
	Profile string
	Bucket  string
}

var teamToData = map[string]teamEnvData{
	"feeds": map[string]environmentData{
		"integration": {
			Region:  "us-west-2",
			Profile: "twitch-feed-dev",
			Bucket:  "twitch-feed-dev",
		},
		"staging": {
			Region:  "us-west-2",
			Profile: "twitch-feed-dev",
			Bucket:  "twitch-feed-dev",
		},
		"canary": {
			Region:  "us-west-2",
			Profile: "twitch-feed-aws",
			Bucket:  "twitch-feed-aws",
		},
		"production": {
			Region:  "us-west-2",
			Profile: "twitch-feed-aws",
			Bucket:  "twitch-feed-aws",
		},
	},
}

type parsedParams struct {
	Team        string
	Service     string
	WhoAmI      string
	TeamEnvData teamEnvData

	verbose bool
	dryrun  bool
}

func (d *deployer) main() {
	if err := d.run(); err != nil {
		fmt.Fprintln(d.errOut, err.Error())
		d.osExit(1)
	}
}

func (d *deployer) writeTemplateToFile(filename string, templateStr string, params interface{}) error {
	if _, err := os.Stat(filename); !os.IsNotExist(err) {
		// Do not write to a file that already exists
		return nil
	}
	f, err := os.Create(filename)
	if err != nil {
		return err
	}
	if err := template.Must(template.New("template_to_file").Parse(templateStr)).Execute(f, params); err != nil {
		return err
	}
	if err := f.Close(); err != nil {
		return err
	}
	return nil
}

func symlinkIfNotExist(old, new string) error {
	if _, err := os.Stat(new); os.IsNotExist(err) {
		return os.Symlink(old, new)
	}
	return nil
}

func (d *deployer) setupEnv(parsedParamsInst *parsedParams, env string) error {
	tmpl := envToTemplate[env]
	if tmpl == "" {
		return errors.New("cannot find template file")
	}
	nameWithTf := parsedParamsInst.Service + ".tff"
	envDir := filepath.Join(".", "terraform", parsedParamsInst.Team, "environments", env, parsedParamsInst.Service)
	if _, err := os.Stat(envDir); os.IsNotExist(err) {
		// Create it then
		if err := os.Mkdir(envDir, 0755); err != nil {
			return err
		}
	}
	if err := symlinkIfNotExist("../common/common.tf", filepath.Join(envDir, "common.tf")); err != nil {
		return err
	}
	if err := symlinkIfNotExist("../../commons/"+nameWithTf, filepath.Join(envDir, nameWithTf)); err != nil {
		return err
	}
	if err := d.writeTemplateToFile(filepath.Join(envDir, "main.tf"), tmpl, parsedParamsInst); err != nil {
		return err
	}
	return nil
}

func (d *deployer) setupTemplates(parsedParamsInst *parsedParams) error {
	moduleDir := filepath.Join(".", "terraform", parsedParamsInst.Team, "modules", parsedParamsInst.Service)
	if _, err := os.Stat(moduleDir); os.IsNotExist(err) {
		// Create it then
		if err := os.Mkdir(moduleDir, 0755); err != nil {
			return err
		}
	}
	if err := d.writeTemplateToFile(filepath.Join(moduleDir, "iam.tf"), defaultIAM, parsedParamsInst); err != nil {
		return err
	}
	if err := d.writeTemplateToFile(filepath.Join(moduleDir, "main.tf"), defaultMain, parsedParamsInst); err != nil {
		return err
	}

	if err := d.writeTemplateToFile(filepath.Join(moduleDir, "varaibles.tf"), defaultVariables, parsedParamsInst); err != nil {
		return err
	}

	commonsDir := filepath.Join(".", "terraform", parsedParamsInst.Team, "environments", "commons")
	if _, err := os.Stat(commonsDir); os.IsNotExist(err) {
		// Should already exists ...
		return err
	}

	if err := d.writeTemplateToFile(filepath.Join(commonsDir, parsedParamsInst.Service+".tff"), defaultCommonsFile, parsedParamsInst); err != nil {
		return err
	}
	return nil
}

func (d *deployer) createModule(parsedParamsInst *parsedParams) error {
	parsedParamsInst.TeamEnvData = teamToData[parsedParamsInst.Team]
	if parsedParamsInst.TeamEnvData == nil {
		return errors.New("unable to find team")
	}
	u, err := user.Current()
	if err != nil {
		return err
	}
	parsedParamsInst.WhoAmI = u.Username
	if err := d.setupTemplates(parsedParamsInst); err != nil {
		return err
	}

	if err := d.setupEnv(parsedParamsInst, "integration"); err != nil {
		return err
	}
	if err := d.setupEnv(parsedParamsInst, "staging"); err != nil {
		return err
	}
	if err := d.setupEnv(parsedParamsInst, "canary"); err != nil {
		return err
	}
	if err := d.setupEnv(parsedParamsInst, "production"); err != nil {
		return err
	}

	return nil
}

func (d *deployer) run() error {
	parsedParamsInst := parsedParams{}
	d.flagSet.StringVar(&parsedParamsInst.Team, "team", "", "aws region to run commands in")
	d.flagSet.StringVar(&parsedParamsInst.Service, "service", "", "aws region to run commands in")
	d.flagSet.BoolVar(&parsedParamsInst.dryrun, "dryrun", false, "Just print out what it would remove, without removing")
	d.flagSet.BoolVar(&parsedParamsInst.verbose, "verbose", false, "Verbose output")
	err := d.flagSet.Parse(d.commandArgs)
	if err != nil {
		return errors.Wrap(err, "Unable to parse command line arguments")
	}

	d.errLog = fmtlogger.NewLogfmtLogger(d.errOut, log.Discard)
	if parsedParamsInst.verbose {
		d.verboseLog = fmtlogger.NewLogfmtLogger(d.errOut, log.Discard)
	} else {
		d.verboseLog = log.Discard
	}

	if err := d.createModule(&parsedParamsInst); err != nil {
		return err
	}
	return nil
}
