package main

const defaultIAM = `
### ----- task level iam policies -----
resource "aws_iam_role" "{{.Service}}_task" {
  name               = "${var.name}-${var.environment}"
  assume_role_policy = "${data.aws_iam_policy_document.{{.Service}}_task.json}"
}

data "aws_iam_policy_document" "{{.Service}}_task" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }
  }
}
`

const defaultMain = `
module "simpleservice" {
  source           = "../simpleservice"
  account          = "${var.account}"
  name             = "${var.name}"
  cluster_name     = "${var.cluster_name}"
  region           = "${var.region}"
  team             = "${var.team}"
  environment      = "${var.environment}"
  desired_count    = "${var.desired_count}"
  target_group_arn = "${var.target_group_arn}"
  task_role_arn    = "${aws_iam_role.{{.Service}}_task.arn}"
  container_port   = "${var.container_port}"
}

module "taskcpuscale" {
  source       = "../taskcpuscale"
  max_size     = "${var.max_size}"
  min_size     = "${var.min_size}"
  account      = "${var.account}"
  region       = "${var.region}"
  cluster_name = "${var.cluster_name}"
  Service      = "${module.simpleservice.service_name}"
  team         = "${var.team}"
  environment  = "${var.environment}"
}

output task_role_arn {
  value = "${aws_iam_role.{{.Service}}_task.arn}"
}

output task_family {
  value = "${module.simpleservice.task_family}"
}

output cluster_name {
  value = "${module.simpleservice.cluster_name}"
}

output service_name {
  value = "${module.simpleservice.service_name}"
}
`

const defaultVariables = `
variable "environment" {}

variable "name" {
  default = "{{.Service}}"
}

variable region {}
variable "team" {}
variable "cluster_name" {}

variable "desired_count" {}
variable "max_size" {}
variable "min_size" {}

variable "target_group_arn" {}
variable "account" {}

variable container_port {}
`

const defaultCommonsFile = `
variable owner {
  default = "{{.WhoAmI}}@twitch.tv"
}

variable container_port {
  default = "8080"
}

variable name {
  default = "{{.Service}}"
}

data "terraform_remote_state" "core" {
  backend = "s3"

  config {
    bucket  = "${var.account}"
    key     = "tfstate/${var.team}/core/terraform/${var.environment}"
    region  = "${var.region}"
    profile = "${var.account}"
  }
}

output task_name {
  value = "${var.name}"
}

output container_port {
  value = "${var.container_port}"
}

output cluster_name {
  value = "${module.{{.Service}}.cluster_name}"
}

output service_name {
  value = "${module.{{.Service}}.service_name}"
}

output task_family {
  value = "${module.{{.Service}}.task_family}"
}

output task_role_arn {
  value = "${module.{{.Service}}.task_role_arn}"
}
`

const defaultIntegrationMain = `
terraform {
  backend "s3" {
    bucket  = "{{.TeamEnvData.integration.Bucket}}"
    key     = "tfstate/{{.Team}}/{{.Service}}/terraform/integration"
    region  = "{{.TeamEnvData.Region}}"
    profile = "{{.TeamEnvData.integration.Profile}}"
  }
}

module "alb" {
  source      = "../../../modules/alb"
  name        = "${var.name}"
  owner       = "${var.owner}"
  environment = "${var.environment}"
  account     = "${var.account}"
  team        = "${var.team}"
}

module "{{.Service}}" {
  source           = "../../../modules/{{.Service}}"
  team             = "${var.team}"
  environment      = "${var.environment}"
  target_group_arn = "${module.alb.target_group_arn}"
  region           = "${var.region}"
  cluster_name     = "${data.terraform_remote_state.core.common_cluster_name}"
  account          = "${var.account}"

  min_size       = "1"
  desired_count  = "2"
  max_size       = "3"
  container_port = "${var.container_port}"
}

output task_cpu {
  value = "1024"
}

output task_mem {
  value = "512"
}
`

const defaultStagingMain = `
terraform {
  backend "s3" {
    bucket  = "{{.TeamEnvData.staging.Bucket}}"
    key     = "tfstate/{{.Team}}/{{.Service}}/terraform/staging"
    region  = "{{.TeamEnvData.Region}}"
    profile = "{{.TeamEnvData.staging.Profile}}"
  }
}

module "alb" {
  source      = "../../../modules/alb"
  name        = "${var.name}"
  owner       = "${var.owner}"
  environment = "${var.environment}"
  account     = "${var.account}"
  team        = "${var.team}"
}

module "{{.Service}}" {
  source           = "../../../modules/{{.Service}}"
  team             = "${var.team}"
  environment      = "${var.environment}"
  target_group_arn = "${module.alb.target_group_arn}"
  region           = "${var.region}"
  cluster_name     = "${data.terraform_remote_state.core.common_cluster_name}"
  account          = "${var.account}"

  min_size       = "1"
  desired_count  = "2"
  max_size       = "3"
  container_port = "${var.container_port}"
}

output task_cpu {
  value = "1024"
}

output task_mem {
  value = "512"
}
`

const defaultCanaryMain = `
terraform {
  backend "s3" {
    bucket  = "{{.TeamEnvData.canary.Bucket}}"
    key     = "tfstate/{{.Team}}/{{.Service}}/terraform/canary"
    region  = "{{.TeamEnvData.Region}}"
    profile = "{{.TeamEnvData.canary.Profile}}"
  }
}

data "terraform_remote_state" "{{.Service}}_production" {
  backend = "s3"

  config {
    bucket  = "${var.account}"
    key     = "tfstate/${var.team}/${var.name}/terraform/production"
    region  = "${var.region}"
    profile = "${var.account}"
  }
}

module "helloworld" {
  source           = "../../../modules/{{.Service}}"
  team             = "${var.team}"
  environment      = "${var.environment}"
  target_group_arn = "${data.terraform_remote_state.{{.Service}}_production.target_group_arn}"
  region           = "${var.region}"
  cluster_name     = "${data.terraform_remote_state.core.common_cluster_name}"
  account          = "${var.account}"

  min_size       = "1"
  desired_count  = "1"
  max_size       = "2"
  container_port = "${var.container_port}"
}

output task_cpu {
  value = "1024"
}

output task_mem {
  value = "512"
}
`

const defaultProductionMain = `
terraform {
  backend "s3" {
    bucket  = "{{.TeamEnvData.production.Bucket}}"
    key     = "tfstate/{{.Team}}/{{.Service}}/terraform/production"
    region  = "{{.TeamEnvData.Region}}"
    profile = "{{.TeamEnvData.production.Profile}}"
  }
}

module "alb" {
  source      = "../../../modules/alb"
  name        = "${var.name}"
  owner       = "${var.owner}"
  environment = "${var.environment}"
  account     = "${var.account}"
  team        = "${var.team}"
}

module "{{.Service}}" {
  source           = "../../../modules/{{.Service}}"
  team             = "${var.team}"
  environment      = "${var.environment}"
  target_group_arn = "${module.alb.target_group_arn}"
  region           = "${var.region}"
  cluster_name     = "${data.terraform_remote_state.core.common_cluster_name}"
  account          = "${var.account}"

  min_size       = "1"
  desired_count  = "2"
  max_size       = "3"
  container_port = "${var.container_port}"
}

output task_cpu {
  value = "1024"
}

output task_mem {
  value = "512"
}

output target_group_arn {
  value = "${module.alb.target_group_arn}"
}
`

var envToTemplate = map[string]string{
	"integration": defaultIntegrationMain,
	"staging":     defaultStagingMain,
	"canary":      defaultCanaryMain,
	"production":  defaultProductionMain,
}
