package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/hashicorp/hcl"
	"github.com/hashicorp/terraform/terraform"
)

type deployer struct {
	flagSet     *flag.FlagSet
	commandArgs []string
	errOut      io.Writer
	out         io.Writer
	osExit      func(int)
}

var instance = deployer{
	flagSet:     flag.NewFlagSet(os.Args[0], flag.ExitOnError),
	commandArgs: os.Args[1:],
	errOut:      os.Stderr,
	osExit:      os.Exit,
	out:         os.Stdout,
}

func main() {
	instance.flagSet.Usage = func() {
		fmt.Println("Creates a new task definition from a Go template file")
		instance.flagSet.PrintDefaults()
	}
	instance.run()
}

type parsedParams struct {
	region    string
	awsRole   string
	bucket    string
	key       string
	verbose   bool
	terraform string
}

func (d *deployer) stringFromMap(m interface{}, keys ...string) string {
	if len(keys) == 0 {
		ret, ok := m.(string)
		if ok {
			return ret
		}
		return ""
	}
	b, ok := m.([]interface{})
	if ok {
		for _, item := range b {
			a := d.stringFromMap(item, keys...)
			if a != "" {
				return a
			}
		}
		return ""
	}
	a, ok := m.(map[string]interface{})
	if !ok {
		return ""
	}
	item := a[keys[0]]
	if item == nil {
		return ""
	}
	return d.stringFromMap(item, keys[1:]...)
}

func (d *deployer) populateParsedFromTerraform(p *parsedParams) error {
	if p.terraform == "" {
		return nil
	}
	input, err := ioutil.ReadFile(p.terraform)
	if err != nil {
		return err
	}
	var v interface{}
	if err2 := hcl.Unmarshal(input, &v); err != nil {
		return err2
	}
	asJSON, err := json.Marshal(v)
	if err != nil {
		return err
	}
	var m map[string]interface{}
	if err := json.Unmarshal(asJSON, &m); err != nil {
		return err
	}
	if bucket := d.stringFromMap(m, "terraform", "backend", "s3", "bucket"); bucket != "" {
		p.bucket = bucket
	}
	if key := d.stringFromMap(m, "terraform", "backend", "s3", "key"); key != "" {
		p.key = key
	}
	if region := d.stringFromMap(m, "terraform", "backend", "s3", "region"); region != "" {
		p.region = region
	}
	return nil
}

func (d *deployer) run() {
	parsedParamsInst := parsedParams{}
	d.flagSet.StringVar(&parsedParamsInst.region, "region", "", "aws region to run commands in")
	d.flagSet.StringVar(&parsedParamsInst.awsRole, "assume_role", "", "An AWS role to assume")
	d.flagSet.StringVar(&parsedParamsInst.key, "key", "", "Bucket key name")
	d.flagSet.StringVar(&parsedParamsInst.bucket, "bucket", "", "Bucket name")
	d.flagSet.StringVar(&parsedParamsInst.terraform, "terraform", "", "terraform file to read region/key/bucket from.")

	d.flagSet.BoolVar(&parsedParamsInst.verbose, "verbose", false, "verbose output")

	err := d.flagSet.Parse(d.commandArgs)
	if err != nil {
		fmt.Fprintf(d.errOut, "Unable to parse command line arguments: %s\n", err.Error())
		d.osExit(1)
		return
	}

	if err = d.populateParsedFromTerraform(&parsedParamsInst); err != nil {
		fmt.Fprintf(d.errOut, "Unable to parsed params: %s\n", err.Error())
		d.osExit(1)
		return
	}

	configs := []*aws.Config{}
	configs = append(configs, &aws.Config{
		Region: &parsedParamsInst.region,
	})
	if parsedParamsInst.awsRole != "" {
		awsSessionForRole, err2 := session.NewSession(configs...)
		if err2 != nil {
			fmt.Fprintf(d.errOut, "Unable to create aws session for assumed role: %s\n", err2.Error())
			d.osExit(1)
			return
		}
		stsclient := sts.New(awsSessionForRole)
		arp := &stscreds.AssumeRoleProvider{
			ExpiryWindow: 10 * time.Second,
			RoleARN:      parsedParamsInst.awsRole,
			Client:       stsclient,
		}
		credentials := credentials.NewCredentials(arp)
		configs = append(configs, &aws.Config{
			Credentials: credentials,
		})
	}

	awsSession, err := session.NewSession(configs...)
	if err != nil {
		fmt.Fprintf(d.errOut, "Unable to create aws session: %s\n", err.Error())
		d.osExit(1)
		return
	}
	backend := s3.New(awsSession, configs...)
	out, err := backend.GetObject(&s3.GetObjectInput{
		Bucket: &parsedParamsInst.bucket,
		Key:    &parsedParamsInst.key,
	})

	if err != nil {
		fmt.Fprintf(d.errOut, "Unable to create aws s3 client: %s\n", err.Error())
		d.osExit(1)
		return
	}
	state := terraform.State{}
	if err := json.NewDecoder(out.Body).Decode(&state); err != nil {
		fmt.Fprintf(d.errOut, "Unable to decode terraform state: %s\n", err.Error())
		d.osExit(1)
		return
	}
	jsonOut := make(map[string]string, len(state.RootModule().Outputs))
	for k, v := range state.RootModule().Outputs {
		jsonOut[k], _ = v.Value.(string)
	}
	if err := json.NewEncoder(d.out).Encode(jsonOut); err != nil {
		fmt.Fprintf(d.errOut, "Unable to decode out JSON result: %s\n", err.Error())
		d.osExit(1)
		return
	}
}
