package main

import (
	"encoding/json"
	"flag"
	"fmt"
	"io"
	"log"
	"os"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/ecs"
	"github.com/aws/aws-sdk-go/service/sts"
)

type deployer struct {
	flagSet     *flag.FlagSet
	commandArgs []string
	errOut      io.Writer
	out         io.Writer
	osExit      func(int)
}

var instance = deployer{
	flagSet:     flag.NewFlagSet(os.Args[0], flag.ExitOnError),
	commandArgs: os.Args[1:],
	errOut:      os.Stderr,
	osExit:      os.Exit,
	out:         os.Stdout,
}

func main() {
	instance.flagSet.Usage = func() {
		fmt.Println("Creates a new task definition from a Go template file")
		instance.flagSet.PrintDefaults()
	}
	instance.run()
}

type parsedParams struct {
	region  string
	cluster string
	service string
	awsRole string
	verbose bool
}

func (d *deployer) run() {
	parsedParamsInst := parsedParams{}
	d.flagSet.StringVar(&parsedParamsInst.region, "region", "", "aws region to run commands in")
	d.flagSet.StringVar(&parsedParamsInst.cluster, "cluster", "", "full ARN of cluster to update")
	d.flagSet.StringVar(&parsedParamsInst.service, "service", "", "service to update")
	d.flagSet.StringVar(&parsedParamsInst.awsRole, "assume_role", "", "An AWS role to assume")

	d.flagSet.BoolVar(&parsedParamsInst.verbose, "verbose", false, "verbose output")

	err := d.flagSet.Parse(d.commandArgs)
	if err != nil {
		fmt.Fprintf(d.errOut, "Unable to parse command line arguments: %s\n", err.Error())
		d.osExit(1)
		return
	}

	// https://gist.github.com/dukejones/36128decdb1e003ac5d77f5c4523f1f5
	// https://github.com/aws/aws-cli/issues/1390
	abc := ecs.DescribeServicesInput{
		Cluster:  &parsedParamsInst.cluster,
		Services: []*string{&parsedParamsInst.service},
	}

	logger := log.New(d.out, "waiter", log.LstdFlags)
	configs := []*aws.Config{}
	configs = append(configs, &aws.Config{
		Region: &parsedParamsInst.region,
		Logger: aws.LoggerFunc(func(v ...interface{}) {
			logger.Println(v...)
		}),
	})
	if parsedParamsInst.awsRole != "" {
		awsSessionForRole, err2 := session.NewSession(configs...)
		if err2 != nil {
			fmt.Fprintf(d.errOut, "Unable to create aws session for assumed role: %s\n", err2.Error())
			d.osExit(1)
			return
		}
		stsclient := sts.New(awsSessionForRole)
		arp := &stscreds.AssumeRoleProvider{
			ExpiryWindow: 10 * time.Second,
			RoleARN:      parsedParamsInst.awsRole,
			Client:       stsclient,
		}
		credentials := credentials.NewCredentials(arp)
		configs = append(configs, &aws.Config{
			Credentials: credentials,
		})
	}

	awsSession, err := session.NewSession(configs...)

	if err != nil {
		fmt.Fprintf(d.errOut, "Unable to create aws session: %s\n", err.Error())
		d.osExit(1)
		return
	}
	ecsClient := ecs.New(awsSession)

	// TODO: Don't count per service 1000 task things as not stable
	// TODO: Would be awesome to print some debug information in a background goroutine
	err = ecsClient.WaitUntilServicesStable(&abc)
	if err != nil {
		fmt.Fprintf(d.errOut, "Services never stablized: %s\n", err.Error())
		d.osExit(1)
		return
	}
	if parsedParamsInst.verbose {
		desc, err := ecsClient.DescribeServices(&abc)
		if err == nil {
			_ = json.NewEncoder(d.errOut).Encode(desc)
		}
	}
}
