package taskmgmt

import (
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awsutil"
	"github.com/aws/aws-sdk-go/service/ecs"
)

// CleanupTasks removes unused ECS tasks
type CleanupTasks struct {
	ECSClient   *ecs.ECS
	VerboseLog  log.Logger
	ErrLog      log.Logger
	TasksToKeep int
	Dryrun      bool
}

func (c *CleanupTasks) verboseLog(vals ...interface{}) {
	if c.VerboseLog != nil {
		c.VerboseLog.Log(vals...)
	}
}

func (c *CleanupTasks) tasksToKeep() int {
	if c.TasksToKeep == 0 {
		return 20
	}
	return c.TasksToKeep
}

func (c *CleanupTasks) errLog(vals ...interface{}) {
	if c.ErrLog != nil {
		c.ErrLog.Log(vals...)
	} else {
		c.verboseLog(vals...)
	}
}

func (c *CleanupTasks) loadUsedTaskDefinitions(allClusters []*string) (map[string]struct{}, error) {
	usedTaskDefinitions := make(map[string]struct{})
	for _, clusterARN := range allClusters {
		if err := c.ECSClient.ListServicesPages(&ecs.ListServicesInput{Cluster: clusterARN}, func(out *ecs.ListServicesOutput, b bool) bool {
			for _, serviceArn := range out.ServiceArns {
				c.verboseLog("describing_service", *serviceArn)
				descOut, err := c.ECSClient.DescribeServices(&ecs.DescribeServicesInput{Services: []*string{serviceArn}, Cluster: clusterARN})
				if err != nil {
					c.errLog("unable to describe service definitions")
					continue
				}
				for _, service := range descOut.Services {
					c.verboseLog("adding task definition", *service.TaskDefinition)
					usedTaskDefinitions[*service.TaskDefinition] = struct{}{}
				}
			}
			return true
		}); err != nil {
			return nil, errors.Wrap(err, "unable to list ecs services")
		}
	}
	return usedTaskDefinitions, nil
}

// Run cleanup on a task family
func (c *CleanupTasks) Run(taskFamily string) ([]string, error) {
	allClusters := make([]*string, 0, 12)
	if err := c.ECSClient.ListClustersPages(&ecs.ListClustersInput{}, func(out *ecs.ListClustersOutput, b bool) bool {
		allClusters = append(allClusters, out.ClusterArns...)
		return true
	}); err != nil {
		return nil, errors.Wrap(err, "unable to list ecs clusters")
	}
	c.verboseLog("known_clusters", awsutil.Prettify(allClusters))

	usedTaskDefinitions, err := c.loadUsedTaskDefinitions(allClusters)
	if err != nil {
		return nil, err
	}

	c.verboseLog("known_task_defs", awsutil.Prettify(usedTaskDefinitions))

	removedTasks := make([]string, 0, 12)
	taskCount := 0
	if err := c.ECSClient.ListTaskDefinitionsPages(&ecs.ListTaskDefinitionsInput{FamilyPrefix: &taskFamily, Sort: aws.String("DESC")}, func(out *ecs.ListTaskDefinitionsOutput, b bool) bool {
		for _, taskDef := range out.TaskDefinitionArns {
			if _, exists := usedTaskDefinitions[*taskDef]; exists {
				continue
			}
			taskCount++
			if taskCount <= c.tasksToKeep() {
				continue
			}
			if c.Dryrun {
				removedTasks = append(removedTasks, *taskDef)
				continue
			}
			_, err := c.ECSClient.DeregisterTaskDefinition(&ecs.DeregisterTaskDefinitionInput{
				TaskDefinition: taskDef,
			})
			if err != nil {
				c.verboseLog("Unable to deregister task", *taskDef)
				continue
			}
			removedTasks = append(removedTasks, *taskDef)
		}
		return true
	}); err != nil {
		return nil, errors.Wrap(err, "Unable to list task definition pages")
	}
	return removedTasks, nil
}
