package sandyconf

import (
	"fmt"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/systems/sandstorm/manager"
)

// Sandyconf is a distconf struct for loading configuration from sandstorm.  By default, it forces users into
// a pattern of Team/Service/Environment to fetch configuration.
type Sandyconf struct {
	Team        string
	Service     string
	Environment string
	Manager     *manager.Manager `nilcheck:"nodepth"`

	distconf.ReaderCache
}

// Get loads a key from sandstorm if it exists
func (s *Sandyconf) Get(key string) ([]byte, error) {
	// TODO: Add timeout (?)
	sec, err := s.Manager.Get(s.keyName(key))
	if err != nil {
		return nil, err
	}
	if sec == nil {
		distconf.ReaderCacheNotify(&s.ReaderCache, key, nil)
		return nil, nil
	}
	distconf.ReaderCacheNotify(&s.ReaderCache, key, sec.Plaintext)
	return sec.Plaintext, nil
}

// Write updates a key in sandstorm.  Fails if you don't have write permission.
func (s *Sandyconf) Write(key string, value []byte) error {
	if value == nil {
		return s.Manager.Delete(s.keyName(key))
	}
	return s.Manager.Put(&manager.Secret{
		Name:      s.keyName(key),
		Plaintext: value,
	})
}

func (s *Sandyconf) keyName(key string) string {
	return fmt.Sprintf("%s/%s/%s/%s", s.Team, s.Service, s.Environment, key)
}

// Close does nothing and exists to satisfy an interface
func (s *Sandyconf) Close() {
}

// Refresh reloads all previously fetched Get keys
func (s *Sandyconf) Refresh() {
	distconf.ReaderCacheRefresh(&s.ReaderCache, s, nil)
}
