package xray

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/aws/request"
)

func (x *XRay) buildHandler() request.NamedHandler {
	return request.NamedHandler{
		Name: "XRayBuildHandler",
		Fn: func(r *request.Request) {
			ctx := r.HTTPRequest.Context()

			ctx, subseg := x.newSegment(ctx, r.ClientInfo.ServiceName)
			r.HTTPRequest = r.HTTPRequest.WithContext(ctx)

			r.HTTPRequest.Header.Set("x-amzn-trace-id", fmt.Sprintf("Root=%s; Parent=%s; Sampled=%d", subseg.root().TraceID, subseg.ID, btoi(subseg.root().sampled)))
		},
	}
}

func (x *XRay) completeHandler() request.NamedHandler {
	return request.NamedHandler{
		Name: "XRayCompleteHandler",
		Fn: func(r *request.Request) {
			subseg := getSegment(r.HTTPRequest.Context())

			subseg.Lock()
			subseg.Namespace = "AWS"

			subseg.aws().Region = r.ClientInfo.SigningRegion
			subseg.aws().Operation = r.Operation.Name
			subseg.aws().RequestID = r.RequestID
			subseg.aws().Retries = r.RetryCount
			if x.optionalMetadata.DynamoDBTableDetector != nil {
				dynamoTableName := x.optionalMetadata.DynamoDBTableDetector(r)
				if dynamoTableName != "" {
					subseg.aws().TableName = dynamoTableName
				}
			}

			subseg.http().response().Status = r.HTTPResponse.StatusCode
			subseg.http().response().ContentLength = int(r.HTTPResponse.ContentLength)
			subseg.Unlock()

			subseg.close(r.Error, x)
		},
	}
}

// AWS adds XRay tracing to an AWS client
func (x *XRay) AWS(c *client.Client) {
	c.Handlers.Build.PushBackNamed(x.buildHandler())
	c.Handlers.Complete.PushBackNamed(x.completeHandler())
}
