package xray

import (
	"context"
	"fmt"
	"testing"

	"code.justin.tv/feeds/xray/ext/pkgerrors"
	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
)

func TestError(t *testing.T) {
	t.Parallel()
	err := stackedError("Test")
	stack := convertStack(err.StackTrace())

	assert.Equal(t, "Test", err.Error())
	assert.Equal(t, "Error", err.Type)
	assert.Equal(t, "TestError", stack[0].Label)
}

func TestErrorf(t *testing.T) {
	t.Parallel()
	err := stackedError("Test")
	stack := convertStack(err.StackTrace())

	assert.Equal(t, "Test", err.Error())
	assert.Equal(t, "Error", err.Type)
	assert.Equal(t, "TestErrorf", stack[0].Label)
}

func TestPanic(t *testing.T) {
	t.Parallel()
	var err *xrayError
	func() {
		defer func() {
			err = panicErr(recover().(string))
		}()
		panic("Test")
	}()
	stack := convertStack(err.StackTrace())

	assert.Equal(t, "Test", err.Error())
	assert.Equal(t, "Panic", err.Type)
	assert.Equal(t, "TestPanic.func1", stack[0].Label)
	assert.Equal(t, "TestPanic", stack[1].Label)
}

func TestPanicf(t *testing.T) {
	t.Parallel()
	var err *xrayError
	func() {
		defer func() {
			err = panicf("%v", recover())
		}()
		panic("Test")
	}()
	stack := convertStack(err.StackTrace())

	assert.Equal(t, "Test", err.Error())
	assert.Equal(t, "Panic", err.Type)
	assert.Equal(t, "TestPanicf.func1", stack[0].Label)
	assert.Equal(t, "TestPanicf", stack[1].Label)
}

func TestException(t *testing.T) {
	t.Parallel()
	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)
	assert.Error(t, x.Capture(context.Background(), "Exception", func(ctx context.Context) error {
		return func() error {
			return fmt.Errorf("raw Error")
		}()
	}))

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	ex := s.Cause.Exceptions[0]
	assert.Equal(t, "raw Error", ex.Message)
	assert.Equal(t, "Error", ex.Type)
	assert.Contains(t, ex.Stack[0].Label, "Capture")
	assert.Contains(t, ex.Stack[0].Path, "/xray/capture.go")
	assert.Equal(t, "TestException", ex.Stack[1].Label)
	assert.Contains(t, ex.Stack[1].Path, "/xray/exception_test.go")
}

func TestExceptionWithStack(t *testing.T) {
	t.Parallel()
	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)
	x.ErrStacks = append(x.ErrStacks, &pkgerrors.StackExtractor{})
	assert.Error(t, x.Capture(context.Background(), "Exception", func(ctx context.Context) error {
		return func() error {
			return errors.New("traced Error")
		}()
	}))

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	ex := s.Cause.Exceptions[0]
	assert.Equal(t, "traced Error", ex.Message)
	assert.Equal(t, "Error", ex.Type)
	assert.Equal(t, "TestExceptionWithStack.func1.1", ex.Stack[0].Label)
	assert.Contains(t, ex.Stack[0].Path, "/xray/exception_test.go")
	assert.Equal(t, "TestExceptionWithStack.func1", ex.Stack[1].Label)
	assert.Contains(t, ex.Stack[1].Path, "/xray/exception_test.go")
	assert.Contains(t, ex.Stack[2].Label, "Capture")
	assert.Contains(t, ex.Stack[2].Path, "/xray/capture.go")
	assert.Equal(t, "TestExceptionWithStack", ex.Stack[3].Label)
	assert.Contains(t, ex.Stack[3].Path, "/xray/exception_test.go")
}

func TestExceptionPanic(t *testing.T) {
	t.Parallel()
	x, testDaemon := makeTestDaemon(t, Config{})
	defer testDaemon.Close(t)
	func() {
		defer func() {
			assert.NotNil(t, recover())
		}()

		assert.Error(t, x.Capture(context.Background(), "Exception", func(ctx context.Context) error {
			return func() error {
				panic("Panic")
			}()
		}))
	}()

	s, e := testDaemon.Recv()
	assert.NoError(t, e)

	ex := s.Cause.Exceptions[0]
	assert.Equal(t, "Panic", ex.Message)
	assert.Equal(t, "Panic", ex.Type)
	assert.Equal(t, "TestExceptionPanic.func1.2.1", ex.Stack[0].Label)
	assert.Contains(t, ex.Stack[0].Path, "/xray/exception_test.go")
	assert.Equal(t, "TestExceptionPanic.func1.2", ex.Stack[1].Label)
	assert.Contains(t, ex.Stack[1].Path, "/xray/exception_test.go")
	assert.Contains(t, ex.Stack[2].Label, "Capture")
	assert.Contains(t, ex.Stack[2].Path, "/xray/capture.go")
	assert.Equal(t, "TestExceptionPanic.func1", ex.Stack[3].Label)
	assert.Contains(t, ex.Stack[3].Path, "/xray/exception_test.go")
	assert.Equal(t, "TestExceptionPanic", ex.Stack[4].Label)
	assert.Contains(t, ex.Stack[4].Path, "/xray/exception_test.go")
}
