package xray

import (
	"context"
	"encoding/json"
	"errors"
	"net"
	"net/http"
	"strconv"
	"sync/atomic"
	"testing"
	"time"
)

func makeTestDaemon(t *testing.T, conf Config) (*XRay, *testdaemon) {
	x := XRay{}
	ret := testdaemon{
		onRunFinished: make(chan struct{}),
		Channel:       make(chan *result, 200),
	}

	if conf.DaemonAddr == "" {
		pickAPortForMe := &net.UDPAddr{
			IP:   net.IPv4(127, 0, 0, 1),
			Port: 0,
		}
		conn2, err := net.ListenUDP("udp", pickAPortForMe)
		if err != nil {
			t.Errorf("Unable to get a free UDP port %s", err)
			return nil, nil
		}
		conf.DaemonAddr = conn2.LocalAddr().String()
		ret.Connection = conn2
	} else {
		addr, err := net.ResolveUDPAddr("udp", conf.DaemonAddr)
		if err != nil {
			t.Errorf("Unable to resolve config UDP addr %s", err)
			return nil, nil
		}
		conn2, err := net.ListenUDP("udp", addr)
		if err != nil {
			t.Errorf("Unable to listen to given UDP addr %s", err)
			return nil, nil
		}
		ret.Connection = conn2
	}
	if err := x.Configure(conf); err != nil {
		t.Errorf("unable to configure xray %s", err)
	}

	// Note: Call close to exit this goroutine
	go ret.Run()
	return &x, &ret
}

type testdaemon struct {
	Connection    *net.UDPConn
	Channel       chan *result
	Done          int32
	onRunFinished chan struct{}
}

type result struct {
	Segment *segment
	Error   error
}

func (td *testdaemon) Close(t *testing.T) {
	atomic.StoreInt32(&td.Done, 1)
	err := td.Connection.Close()
	<-td.onRunFinished
	if err != nil {
		t.Errorf("unable to finish closing testing daemon %s", err.Error())
	}
}

func (td *testdaemon) Run() {
	buffer := make([]byte, maxSegmentSize)
	for atomic.LoadInt32(&td.Done) == 0 {
		n, _, err := td.Connection.ReadFromUDP(buffer)
		if err != nil {
			td.Channel <- &result{nil, err}
			continue
		}

		buffered := buffer[len(header):n]

		seg := &segment{}
		err = json.Unmarshal(buffered, &seg)
		if err != nil {
			td.Channel <- &result{nil, err}
			continue
		}

		seg.sampled = true
		td.Channel <- &result{seg, err}
	}
	close(td.onRunFinished)
}

func (td *testdaemon) Recv() (*segment, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 100*time.Millisecond)
	defer cancel()
	select {
	case r := <-td.Channel:
		return r.Segment, r.Error
	case <-ctx.Done():
		return nil, ctx.Err()
	case <-td.onRunFinished:
		return nil, errors.New("testing daemon closed")
	}
}

type xRayHeaders struct {
	RootTraceID string
	ParentID    string
	Sampled     bool
}

func parseHeadersForTest(h http.Header) xRayHeaders {
	m := parseHeaders(h)
	s, _ := strconv.ParseBool(m["Sampled"])

	return xRayHeaders{
		RootTraceID: m["Root"],
		ParentID:    m["Parent"],
		Sampled:     s,
	}
}
