package heartbeat

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/signer/v4"
)

const endpointHeartbeat = "heartbeat"
const defaultURL = "https://zhpqn2llm0.execute-api.us-west-2.amazonaws.com/production"
const httpTimeout = 30
const defaultHost = "UnknownHost"
const defaultService = "UnknownService"
const httpHeaderContentType = "Content-Type"
const headerContentTypeJSON = "application/json"

var defaultInterval = 60 * time.Second

//Config for inventory client
type Config struct {
	Interval time.Duration
	URL      string
	Service  string
	Host     string
	Region   string
}

func nopLogger() *logrus.Logger {
	return &logrus.Logger{
		Out: ioutil.Discard,
	}
}

// HTTPClient so that we can mock http.client in tests.
type HTTPClient interface {
	Do(req *http.Request) (*http.Response, error)
}

//Client for inventory
type Client struct {
	httpClient     HTTPClient
	Config         *Config
	heartbeatState *heartbeatState
	Logger         *logrus.Logger
	cancel         chan struct{}
	cancelConfirm  chan struct{}
}

// API is the interface that Client implements
type API interface {
	UpdateHeartbeat(secret *Secret)
	Start()
	SendHeartbeat()
	Stop() error
}

// New condfigure a new http client, the client will send the secret status to the
// inventory server periodically.
// Once the oject is created the caller must call Start to start sending
// heartbeatState to inventory
func New(credentials *credentials.Credentials, config *Config, logger *logrus.Logger) (client *Client) {
	config = buildConfig(config)

	secretToReport := make(map[string]*Secret)
	heartbeatState := &heartbeatState{
		secretsToReport: secretToReport,
		host:            config.Host,
		service:         config.Service,
	}

	if logger == nil {
		logger = nopLogger()
	}

	client = &Client{
		httpClient: &http.Client{
			Transport: &sigV4Roundtripper{
				inner:  http.DefaultTransport,
				signer: v4.NewSigner(credentials),
				region: config.Region,
			},
		},
		heartbeatState: heartbeatState,
		Logger:         logger,
		Config:         config,
		cancel:         make(chan struct{}),
		cancelConfirm:  make(chan struct{}),
	}

	return
}

func defaultConfig() (config *Config) {

	config = &Config{
		Region:   "us-west-2",
		URL:      defaultURL,
		Interval: defaultInterval,
		Host:     hostName(),
		Service:  serviceName(),
	}
	return
}

func buildConfig(providedConfig *Config) (config *Config) {
	config = providedConfig
	defaultConfig := defaultConfig()
	if providedConfig == nil {
		config = defaultConfig
		return
	}

	if config.URL == "" {
		config.URL = defaultConfig.URL
	}
	if config.Interval == 0 {
		config.Interval = defaultConfig.Interval
	}
	if config.Service == "" {
		config.Service = defaultConfig.Service
	}
	if config.Host == "" {
		config.Host = defaultConfig.Host
	}
	if config.Region == "" {
		config.Region = defaultConfig.Region
	}

	return
}

// UpdateHeartbeat updates the current state of heartbeats to send to
// the inventory server.
func (client *Client) UpdateHeartbeat(secret *Secret) {
	if secret == nil {
		return
	}
	client.Logger.Debugf("updating heartbeat for secret: %s", secret.Name)
	secret.FetchedAt = time.Now().Unix()
	client.heartbeatState.secretsToReport[secret.Name] = secret
	return
}

// Start sends the current state of heartbeats to inventory server
func (client *Client) Start() {
	signals := make(chan os.Signal)

	signal.Notify(signals, syscall.SIGINT, syscall.SIGTERM)

	client.Logger.Debugf(
		"reporting secret status to inventory at '%s' every %d seconds",
		client.Config.URL,
		client.Config.Interval/time.Second,
	)
	for {
		timer := time.NewTimer(client.Config.Interval)
		select {
		case <-signals:
			client.Logger.Infof("Flushing one last time due to os signal exit")
			client.SendHeartbeat()
			return
		case <-client.cancel:
			client.Logger.Infof("Flushing one last time due to cancel request")
			client.SendHeartbeat()
			client.sendCancelConfirm()
			return
		case <-timer.C:
			// send the heartbeats to server and sleep for specified interval
			client.SendHeartbeat()
			client.Logger.Debugf("Sleeping for %v seconds", client.Config.Interval)
		}
	}
}

func (client *Client) sendCancelConfirm() {
	timer := time.NewTimer(client.Config.Interval)
	select {
	case client.cancelConfirm <- struct{}{}:
		return
	case <-timer.C:
		client.Logger.Warnf("timeout when sending cancel confirmation")
		return
	}
}

// Stop stops the reporter goroutine
func (client *Client) Stop() (err error) {
	timer := time.NewTimer(10 * time.Second)
	err = client.sendStopRequest(timer)
	if err != nil {
		return
	}
	err = client.receiveCancelConfirm(timer)
	return
}

func (client *Client) sendStopRequest(timer *time.Timer) (err error) {
	select {
	case client.cancel <- struct{}{}:
		return
	case <-timer.C:
		err = errors.New("could not stop reporter gorountine")
		return
	}
}

func (client *Client) receiveCancelConfirm(timer *time.Timer) (err error) {
	select {
	case <-client.cancelConfirm:
		return
	case <-timer.C:
		err = errors.New("timeout when receiving cancel confirmation")
		return
	}
}

// SendHeartbeat to manually flush the heartbeat. usually called in Start goroutine.
func (client *Client) SendHeartbeat() {
	if len(client.heartbeatState.secretsToReport) == 0 {
		client.Logger.Debugln("Nothing to report.")
		return
	}
	err := client.putHeartbeat()
	if err != nil {
		client.Logger.Warnf("failed to send heartbeat to inventory: err: %s", err.Error())
	}
	return
}

//Send the current heart state to inventory server
func (client *Client) putHeartbeat() (err error) {

	client.Logger.Debugf("Reporting heartbeat for %d secrets to %s. ", len(client.heartbeatState.secretsToReport), client.Config.URL)
	req, err := client.buildPutHeartbeatRequest()
	if err != nil {
		err = fmt.Errorf("Failed to build heartbeat request. Error: %s", err.Error())
		return
	}

	resp, err := client.httpClient.Do(req)

	if err != nil {
		err = fmt.Errorf("error sending request to inventory %s. err: %s", client.Config.URL, err.Error())
		return
	}

	defer func() {
		closeErr := resp.Body.Close()
		if closeErr != nil && err == nil {
			err = closeErr
		}
		return
	}()

	switch resp.StatusCode {
	case http.StatusOK:
		client.Logger.Debugf("Done reporting.")
	default:
		var body []byte
		body, err = ioutil.ReadAll(resp.Body)
		if err != nil {
			return
		}
		err = fmt.Errorf("unexpected status code %d returned: %s", resp.StatusCode, string(body))
	}
	return
}

func (client *Client) buildPutHeartbeatRequest() (req *http.Request, err error) {

	hb := &heartbeat{}

	hb.Service = client.heartbeatState.service
	hb.Host = client.heartbeatState.host

	//Build secrets to be send over to inventory
	var secrets []*Secret

	// build an inventory.heartbeats to send to the servers.
	for _, secret := range client.heartbeatState.secretsToReport {
		secrets = append(secrets, secret)
	}
	hb.Secrets = secrets

	bs := bytes.NewBuffer(nil)

	err = json.NewEncoder(bs).Encode(hb)
	if err != nil {
		return
	}

	reqBody := bytes.NewReader(bs.Bytes())

	putHeartbeatEndpoint := fmt.Sprintf("%s/%s", client.Config.URL, endpointHeartbeat)
	req, err = http.NewRequest("PUT", putHeartbeatEndpoint, reqBody)

	req.Header.Set(httpHeaderContentType, headerContentTypeJSON)
	if err != nil {
		return
	}

	return
}

//serviceName return processName. if non found return unknownService
func serviceName() (service string) {
	service = os.Args[0]
	if service == "" {
		service = defaultService
	}
	return
}

//hostName return hostname of the machine. if non found return unknownHost
func hostName() (host string) {
	host, err := os.Hostname()
	if err != nil {
		host = defaultHost
	}
	return
}
