package manager

import (
	"os"
	"time"

	"code.justin.tv/common/ddbmetrics"
	"code.justin.tv/systems/sandstorm/inventory/heartbeat"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-sdk-go/service/kms"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/sirupsen/logrus"
)

// API describes the manager interface
type API interface {
	AuditTableName() string
	CheckKMS() error
	CheckTable() error
	Copy(source, destination string) error
	CreateTable() error
	Decrypt(secret *Secret) error
	Delete(secretName string) error
	Exist(secretName string) (bool, error)
	CrossEnvironmentSecretsSet(secretNames []string) (crossEnvSecrets map[string]struct{}, err error)
	Get(secretName string) (*Secret, error)
	GetEncrypted(secretName string) (*Secret, error)
	GetVersion(secretName string, version int64) (*Secret, error)
	GetVersionsEncrypted(secretName string, limit int64, offsetKey int64) (*VersionsPage, error)
	List() ([]*Secret, error)
	ListNamespace(namespace string) ([]*Secret, error)
	ListNamespaces() ([]string, error)
	NamespaceTableName() string
	Patch(patchInput *PatchInput) error
	Post(secret *Secret) error
	Put(secret *Secret) error
	Revert(name string, version int64) error
	Seal(secret *Secret) error
	TableName() string
	CleanUp() error
}

// Manager is the struct that a caller interacts with for managing
// secrets.
type Manager struct {
	Envelope        Enveloper
	DynamoDB        dynamodbiface.DynamoDBAPI
	Config          Config
	metrics         *ddbmetrics.Publisher
	Logger          *logrus.Logger
	inventoryClient heartbeat.API
}

// Config is a struct for configuring the manager
// XXX need to be able to provide an aws.Config for STS credentials,
// region et c
type Config struct {
	// AWS config structure. If left nil, will be replaced by a
	// default config with region us-west-2.
	AWSConfig *aws.Config
	// KMS KeyId - this is either a key alias prefixed by 'alias/',
	// a key ARN or a key UUID
	KeyID string
	// DynamoDB Table name
	TableName string
	// Provisioned Read and Write throughput when creating the
	// table, not used in normal operations. If left to 0 these
	// will be replaced by 1 by default.
	ProvisionedThroughputRead  int64
	ProvisionedThroughputWrite int64
	StatsdHostPort             string
	StatsdPrefix               string
	Logger                     *logrus.Logger

	Host               string
	InventoryRoleARN   string
	InventoryStatusURL string
	InventoryInterval  time.Duration
	ServiceName        string
}

// DefaultConfig returns a config struct with some sane defaults that
// is merged with the provided config in New
func DefaultConfig() Config {
	logger := &logrus.Logger{
		Out:       os.Stderr,
		Formatter: new(logrus.TextFormatter),
		Level:     logrus.WarnLevel,
	}

	return Config{
		ProvisionedThroughputRead:  1,
		ProvisionedThroughputWrite: 1,
		AWSConfig:                  &aws.Config{Region: aws.String("us-west-2")},
		Logger:                     logger,
		InventoryRoleARN:           "arn:aws:iam::854594403332:role/inventory-gateway-execute-api-invoke-production",
	}
}

func mergeConfig(provided *Config, defConfig Config) {
	if provided.AWSConfig == nil {
		provided.AWSConfig = defConfig.AWSConfig
	}
	if provided.ProvisionedThroughputRead == 0 {
		provided.ProvisionedThroughputRead = defConfig.ProvisionedThroughputRead
	}
	if provided.ProvisionedThroughputWrite == 0 {
		provided.ProvisionedThroughputWrite = defConfig.ProvisionedThroughputWrite
	}
	if provided.Logger == nil {
		provided.Logger = defConfig.Logger
	}
	if provided.InventoryRoleARN == "" {
		provided.InventoryRoleARN = defConfig.InventoryRoleARN
	}
}

// New creates a Manager from config. Merges configuration with
// DefaultConfig()
func New(config Config) *Manager {
	var stats statsd.Statter
	defConfig := DefaultConfig()
	mergeConfig(&config, defConfig)

	if config.StatsdHostPort != "" || config.StatsdPrefix != "" {
		hostName, err := os.Hostname()
		if err != nil {
			hostName = "unknown"
		}

		stats, err = statsd.NewClient(config.StatsdHostPort, config.StatsdPrefix+"."+hostName)
		if err != nil {
			config.Logger.Printf("Couldn't start statsd: %s", err.Error())
			stats, err = statsd.NewNoopClient()
			if err != nil {
				// noop client won't return anything else
			}
		}
	} else {
		var err error
		stats, err = statsd.NewNoopClient()
		if err != nil {
			// noop client won't return anything else
		}
	}

	session := session.New(config.AWSConfig)

	ddb := dynamodb.New(session)
	m := ddbmetrics.New(ddb, stats)
	m.Start()

	mgr := &Manager{
		DynamoDB: ddb,
		Config:   config,
		Envelope: &Envelope{
			KMS: kms.New(session),
		},
		metrics: m,
		Logger:  config.Logger,
	}

	mgr.inventoryClient = heartbeat.New(
		stscreds.NewCredentials(session, config.InventoryRoleARN),
		&heartbeat.Config{
			Interval: config.InventoryInterval,
			URL:      config.InventoryStatusURL,
			Service:  config.ServiceName,
			Region:   *config.AWSConfig.Region,
			Host:     config.Host,
		},
		config.Logger,
	)
	go mgr.inventoryClient.Start()

	return mgr

}

// CleanUp cleans exits any go routines spawned
func (mgr *Manager) CleanUp() (err error) {
	err = mgr.inventoryClient.Stop()
	return
}
