package manager

import (
	"errors"
	"io/ioutil"
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/inventory/consumedsecrets"
	"code.justin.tv/systems/sandstorm/testutil"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/sirupsen/logrus"
)

func nopLogger() *logrus.Logger {
	return &logrus.Logger{
		Out: ioutil.Discard,
	}
}

func createTestManager() (*Manager, error) {
	config, err := testutil.LoadTestConfig()
	if err != nil {
		return nil, err
	}
	if config == nil {
		return nil, errors.New("test config is nil")
	}

	awsConfig := &aws.Config{
		Region: aws.String(config.Sandstorm.Region),
	}
	stsclient := sts.New(session.New(awsConfig))
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      config.Sandstorm.RoleArn,
		Client:       stsclient,
	}
	awsConfig.WithCredentials(credentials.NewCredentials(arp))
	return New(Config{
		AWSConfig:          awsConfig,
		TableName:          config.Sandstorm.TableName,
		KeyID:              config.Sandstorm.KeyID,
		Host:               "someHostName",
		InventoryRoleARN:   config.Sandstorm.InventoryRoleARN,
		InventoryStatusURL: config.Sandstorm.InventoryStatusURL,
		InventoryInterval:  1 * time.Second,
		ServiceName:        config.Sandstorm.ServiceName,
		Logger:             nopLogger(),
	}), nil
}

func createTestConsumedSecretsClient(t *testing.T) (api consumedsecrets.API) {
	config, err := testutil.LoadTestConfig()
	if err != nil {
		t.Fatal(err)
	}

	awsConfig := &aws.Config{
		Region: aws.String(config.Sandstorm.Region),
	}

	stsclient := sts.New(session.New(awsConfig))
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      config.Sandstorm.InventoryAdminRoleARN,
		Client:       stsclient,
	}
	awsConfig.WithCredentials(credentials.NewCredentials(arp))

	sess := session.New(awsConfig)

	api = consumedsecrets.New(
		&consumedsecrets.Config{
			TableName: "heartbeats-testing",
		},
		sess,
	)

	return
}
