package manager

import (
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/testutil"

	"github.com/stretchr/testify/assert"
)

func TestManagerPatchIntegration(t *testing.T) {
	// manager with real dynamo
	m, err := createTestManager()
	if err != nil {
		t.Error(err)
	}

	t.Run("patch secret in stages", func(t *testing.T) {
		const originalSecretValue = "myPlaintext"
		const patchedSecretValue = "myPatchedPlaintext"

		secretName := testutil.GetRandomSecretName()
		defer func() {
			err := m.Delete(secretName)
			assert.NoError(t, err)
		}()

		assertCurrentSecret := func(t *testing.T, s *Secret) {
			currentSecret, err := m.Get(secretName)
			assert.NoError(t, err)

			// zero out other fields for comparison
			currentSecret.key = nil
			currentSecret.ciphertext = nil

			assert.Equal(t, s, currentSecret)
		}

		err := m.Post(&Secret{
			Class:     ClassPublic,
			Name:      secretName,
			Plaintext: []byte(originalSecretValue),
		})
		assert.NoError(t, err)

		// update KeyARN and UpdatedAt for test
		currentSecret, err := m.Get(secretName)
		assert.NoError(t, err)
		keyARN := currentSecret.KeyARN
		updatedAt := currentSecret.UpdatedAt

		getOriginalSecret := func() *Secret {
			return &Secret{
				Class:     ClassPublic,
				Name:      secretName,
				Plaintext: []byte(originalSecretValue),
				KeyARN:    keyARN,
				UpdatedAt: updatedAt,
			}
		}

		t.Run("check first value", func(t *testing.T) {
			assertCurrentSecret(t, getOriginalSecret())
		})

		t.Run("patch noop", func(t *testing.T) {
			assert := assert.New(t)

			err := m.Patch(&PatchInput{
				Name: secretName,
			})
			assert.NoError(err)

			assertCurrentSecret(t, getOriginalSecret())
		})

		t.Run("update DoNotBroadcast", func(t *testing.T) {
			assert := assert.New(t)

			dnb := true
			err := m.Patch(&PatchInput{
				Name:           secretName,
				DoNotBroadcast: &dnb,
			})
			assert.NoError(err)

			expectedSecret := getOriginalSecret()
			expectedSecret.DoNotBroadcast = true
			assertCurrentSecret(t, expectedSecret)
		})

		t.Run("update Class", func(t *testing.T) {
			assert := assert.New(t)

			cls := ClassCustomer
			err := m.Patch(&PatchInput{
				Name:  secretName,
				Class: &cls,
			})
			assert.NoError(err)

			expectedSecret := getOriginalSecret()
			expectedSecret.DoNotBroadcast = true
			expectedSecret.Class = ClassCustomer
			assertCurrentSecret(t, expectedSecret)
		})

		t.Run("update CrossEnv", func(t *testing.T) {
			assert := assert.New(t)

			crossEnv := true
			err := m.Patch(&PatchInput{
				Name:     secretName,
				CrossEnv: &crossEnv,
			})
			assert.NoError(err)

			expectedSecret := getOriginalSecret()
			expectedSecret.DoNotBroadcast = true
			expectedSecret.Class = ClassCustomer
			expectedSecret.CrossEnv = true
			assertCurrentSecret(t, expectedSecret)
		})

		t.Run("update Plaintext", func(t *testing.T) {
			assert := assert.New(t)

			// versions have a 1 second delta
			time.Sleep(time.Second)
			err := m.Patch(&PatchInput{
				Name:      secretName,
				Plaintext: []byte(patchedSecretValue),
			})
			assert.NoError(err)

			currentSecret, err := m.Get(secretName)
			assert.NoError(err)
			assert.True(currentSecret.UpdatedAt > updatedAt)

			expectedSecret := getOriginalSecret()
			expectedSecret.Plaintext = []byte(patchedSecretValue)
			expectedSecret.DoNotBroadcast = true
			expectedSecret.Class = ClassCustomer
			expectedSecret.CrossEnv = true
			expectedSecret.UpdatedAt = currentSecret.UpdatedAt
			assertCurrentSecret(t, expectedSecret)
		})
	})
}
