# Creates an ALB and exports the ALB target group

## User warning
## You may need to prepare your terraform twice because of a dependency between the ALB listener and ecs group
# See https://github.com/hashicorp/terraform/issues/12634 for when this is fixed

data "terraform_remote_state" "account" {
  backend = "s3"

  config {
    bucket  = "${var.account}"
    key     = "tfstate/feed/terraform/accounts/${var.account}"
    region  = "us-west-2"
    profile = "${var.account}"
  }
}

resource "aws_alb_target_group" "app" {
  name                 = "${var.name}-${var.environment}"
  deregistration_delay = 30

  # Use a dummy low port because we want applications to give us their port
  port     = "1"
  protocol = "HTTP"
  vpc_id   = "${data.terraform_remote_state.account.vpc_id}"

  health_check {
    path     = "/debug/running"
    protocol = "HTTP"
  }
}

resource "aws_alb_listener" "app" {
  load_balancer_arn = "${aws_alb.app.arn}"
  port              = "80"
  protocol          = "HTTP"

  default_action {
    target_group_arn = "${aws_alb_target_group.app.arn}"
    type             = "forward"
  }
}

resource "aws_alb" "app" {
  name                       = "${var.name}-${var.environment}"
  security_groups            = ["${data.terraform_remote_state.account.twitch_subnets_sg}"]
  subnets                    = ["${split(",", "${data.terraform_remote_state.account.private_subnets}")}"]
  enable_deletion_protection = true
  internal                   = true

  access_logs {
    bucket = "${var.account}-logs"
    prefix = "elb-logs/${var.name}-${var.environment}"
  }

  tags {
    Name        = "${var.name}"
    Environment = "${var.environment}"
    Owner       = "${var.owner}"
    Team        = "${var.team}"
  }
}

output "target_group_arn" {
  value = "${aws_alb_target_group.app.arn}"
}

module "dns" {
  source      = "../dns"
  team        = "${var.team}"
  dns_name    = "${aws_alb.app.dns_name}"
  zone_id     = "${aws_alb.app.zone_id}"
  name        = "${var.name}"
  environment = "${var.environment}"
}

output "dns_name" {
  value = "${aws_alb.app.dns_name}"
}

output "zone_id" {
  value = "${aws_alb.app.zone_id}"
}
