# allow builds
resource "aws_iam_role" "build_deployment" {
  # Note: You will have to run this terraform twice, if you get an -Invalid principal in policy- error
  name               = "${var.environment}-build-deployment"
  assume_role_policy = "${data.aws_iam_policy_document.build_deployment.json}"
  description        = "role for deployment to assume when needing to deploy new versions of code"
}

data "aws_iam_policy_document" "build_deployment" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs.amazonaws.com"]
    }
  }

  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type = "AWS"

      // Ideally this is exported so I can read it from somewhere
      identifiers = ["arn:aws:iam::043714768218:role/jenkins-prod-slave"]
    }
  }

  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type = "AWS"

      // Ideally this is exported so I can read it from somewhere
      identifiers = ["${aws_iam_user.deployment.arn}"]
    }
  }
}

# devtools jenkins won't allow me to assume roles directly and requires aws credentials stored on the box.
# This is a security flaw that I disagree with, but I also cannot maintain a jenkins cluster myself, so I will do as
# they request.
resource "aws_iam_user" "deployment" {
  name = "deploy-${var.environment}"
}

data "aws_iam_policy_document" "allow_build_user_assume_build_role" {
  statement {
    actions   = ["sts:AssumeRole"]
    resources = ["${aws_iam_role.build_deployment.arn}"]
  }
}

resource "aws_iam_user_policy" "allow_user_assume_role" {
  name   = "deploy-user-assume-role-${var.environment}"
  user   = "${aws_iam_user.deployment.name}"
  policy = "${data.aws_iam_policy_document.allow_build_user_assume_build_role.json}"
}

resource "aws_iam_role_policy_attachment" "build_deployment_attach_policyEC2" {
  role       = "${aws_iam_role.build_deployment.name}"
  policy_arn = "arn:aws:iam::aws:policy/AmazonEC2ContainerServiceFullAccess"
}

resource "aws_iam_role_policy_attachment" "build_deployment_attach_policyS3" {
  role       = "${aws_iam_role.build_deployment.name}"
  policy_arn = "arn:aws:iam::aws:policy/AmazonS3ReadOnlyAccess"
}

# Setup IAM to allow services to register w/ the ALB
resource "aws_iam_role" "ecs_service" {
  name               = "${var.environment}-ecs-service"
  assume_role_policy = "${data.aws_iam_policy_document.ecs_service.json}"
}

data "aws_iam_policy_document" "ecs_service" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs.amazonaws.com"]
    }
  }
}

resource "aws_iam_instance_profile" "ecs_service" {
  name = "ecs-service-${var.environment}"
  role = "${aws_iam_role.ecs_service.name}"
}

resource "aws_iam_role_policy_attachment" "ecs_service_attach_policy" {
  role       = "${aws_iam_role.ecs_service.name}"
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceRole"
}

# Setup IAM to allow task autoscaling
resource "aws_iam_role" "ecs_autoscale" {
  name               = "${var.environment}-ecs-task-autoscale"
  assume_role_policy = "${data.aws_iam_policy_document.ecs_autoscale.json}"
}

data "aws_iam_policy_document" "ecs_autoscale" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["application-autoscaling.amazonaws.com"]
    }
  }
}

resource "aws_iam_instance_profile" "ecs_autoscale" {
  name = "ecs-autoscale-${var.environment}"
  role = "${aws_iam_role.ecs_autoscale.name}"
}

resource "aws_iam_role_policy_attachment" "ecs_autoscale_attach_policy" {
  role       = "${aws_iam_role.ecs_autoscale.name}"
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceAutoscaleRole"
}

# Setup IAM to allow hosts to update ECS deployed tasks

resource "aws_iam_role" "ecs_host" {
  name               = "${var.environment}-ecs-host"
  assume_role_policy = "${data.aws_iam_policy_document.ecs_host.json}"
}

data "aws_iam_policy_document" "ecs_host" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

resource "aws_iam_instance_profile" "ecs_host" {
  name = "ecs-host-${var.environment}"
  role = "${aws_iam_role.ecs_host.name}"
}

resource "aws_iam_role_policy_attachment" "ecs_host_attach_policy" {
  role       = "${aws_iam_role.ecs_host.name}"
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2ContainerServiceforEC2Role"
}

# Setup task for statsite

resource "aws_ecs_task_definition" "statsite" {
  family                = "statsite-${var.environment}"
  container_definitions = "${data.template_file.statsite_task_def.rendered}"
  network_mode          = "bridge"
}

data "template_file" "statsite_task_def" {
  template = "${file("${path.module}/statsite_task_definition.json")}"

  vars {
    image    = "docker-registry.internal.justin.tv/statsite"
    region   = "${var.region}"
    env      = "${var.environment}"
    loggroup = "${aws_cloudwatch_log_group.container-logs.name}"
    mem      = 500
    cpu      = 512                                               // Half a core should be enough ...
  }
}

# Setup task for xray
# TODO: Attack correct xray permissions

data "template_file" "xray_task_def" {
  template = "${file("${path.module}/xray_task_definition.json")}"

  vars {
    image    = "docker-registry.internal.justin.tv/xray"
    region   = "${var.region}"
    env      = "${var.environment}"
    loggroup = "${aws_cloudwatch_log_group.container-logs.name}"
    mem      = 500
    cpu      = 512                                               // Half a core should be enough ...
  }
}

### ----- task level iam policies -----
resource "aws_iam_role" "xray_task" {
  name               = "xray-${var.environment}"
  assume_role_policy = "${data.aws_iam_policy_document.xray_task.json}"
}

data "aws_iam_policy_document" "xray_task" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }
  }
}

resource "aws_iam_role_policy" "policy" {
  name = "xray_${var.environment}_assume_role"
  role = "${aws_iam_role.xray_task.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": "sts:AssumeRole",
      "Resource": ["${var.xray_role}"]
    }
  ]
}
EOF
}

resource "aws_ecs_task_definition" "xray" {
  family                = "xray-${var.environment}"
  container_definitions = "${data.template_file.xray_task_def.rendered}"
  network_mode          = "bridge"
  task_role_arn         = "${aws_iam_role.xray_task.arn}"
}

# Setup log group for ECS container logs

resource "aws_cloudwatch_log_group" "container-logs" {
  name              = "container-logs-${var.environment}"
  retention_in_days = "${var.container_log_retention}"

  tags {
    env  = "${var.environment}"
    team = "${var.team}"
  }
}

module "common_cluster" {
  source               = "../ecscluster"
  environment          = "${var.environment}"
  name                 = "common-${var.environment}"
  team                 = "${var.team}"
  xray_task_family     = "${aws_ecs_task_definition.statsite.family}"
  statsite_task_family = "${aws_ecs_task_definition.xray.family}"
}

# Output variables

output "container_loggroup" {
  value = "${aws_cloudwatch_log_group.container-logs.name}"
}

output "container_iam_role" {
  value = "${aws_iam_role.ecs_host.arn}"
}

output "service_iam_role" {
  value = "${aws_iam_role.ecs_service.arn}"
}

output "xray_task_role" {
  value = "${aws_iam_role.xray_task.arn}"
}

output "task_autoscale_iam_role" {
  value = "${aws_iam_role.ecs_autoscale.arn}"
}

output "container_instance_profile_id" {
  value = "${aws_iam_instance_profile.ecs_host.id}"
}

output "statsite_task_family" {
  value = "${aws_ecs_task_definition.statsite.family}"
}

output "xray_task_family" {
  value = "${aws_ecs_task_definition.xray.family}"
}

output common_cluster_name {
  value = "${module.common_cluster.cluster_name}"
}

output common_cluster_id {
  value = "${module.common_cluster.cluster_id}"
}

output build_role {
  value = "${aws_iam_role.build_deployment.arn}"
}
