# Primary application ASG and ELB

data "terraform_remote_state" "account" {
  backend = "s3"

  config {
    bucket  = "${var.account}"
    key     = "tfstate/feed/terraform/accounts/${var.account}"
    region  = "us-west-2"
    profile = "${var.account}"
  }
}

data "terraform_remote_state" "asg_lambda" {
  backend = "s3"

  config {
    bucket  = "${var.account}"
    key     = "tfstate/web/asg-r53-lambda/services/lambda/${var.account}"
    region  = "us-west-2"
    profile = "${var.account}"
  }
}

resource "aws_launch_configuration" "app_conf" {
  name_prefix     = "${var.name}-${var.environment}"
  image_id        = "${var.ami}"
  instance_type   = "${var.instance_type}"
  security_groups = ["${data.terraform_remote_state.account.twitch_subnets_sg}"]
  user_data       = "${data.template_file.app_user_data.rendered}"

  key_name             = "${var.host_key_name}"
  iam_instance_profile = "${var.container_instance_profile_id}"

  root_block_device {
    volume_size = "${var.root_volume_size}"
  }

  lifecycle {
    create_before_destroy = true
  }
}

data "template_file" "app_user_data" {
  template = "${file("${path.module}/asg_puppetizer.template")}"

  vars {
    ecs_cluster_name = "${var.cluster_name}"
    environment      = "${var.environment}"
  }
}

resource "aws_autoscaling_group" "app_asg" {
  name                      = "${var.name}-${var.environment}"
  vpc_zone_identifier       = ["${split(",", "${data.terraform_remote_state.account.private_subnets}")}"]
  max_size                  = "${var.max_size}"
  min_size                  = "${var.min_size}"
  desired_capacity          = "${var.desired_capacity}"
  health_check_grace_period = 600
  health_check_type         = "EC2"
  launch_configuration      = "${aws_launch_configuration.app_conf.name}"

  tag {
    key                 = "Name"
    value               = "${var.name}"
    propagate_at_launch = true
  }

  tag {
    key                 = "Environment"
    value               = "${var.environment}"
    propagate_at_launch = true
  }

  tag {
    key                 = "Owner"
    value               = "${var.owner}"
    propagate_at_launch = true
  }

  tag {
    key                 = "Team"
    value               = "${var.team}"
    propagate_at_launch = true
  }

  // If we don't ignore this, then each rerun of terraform resets the desired capacity if it ever auto scaled
  lifecycle {
    ignore_changes = ["desired_capacity"]
  }
}

resource "aws_autoscaling_policy" "app_asg_scaleout_policy" {
  name                   = "${var.name}-${var.environment}-scaleout-policy"
  scaling_adjustment     = 1
  adjustment_type        = "ChangeInCapacity"
  cooldown               = 1200
  autoscaling_group_name = "${aws_autoscaling_group.app_asg.name}"
}

resource "aws_cloudwatch_metric_alarm" "asg_high_cpu" {
  alarm_name          = "${var.name}-${var.environment}-high-cpu"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = "5"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/EC2"
  period              = "60"
  statistic           = "Average"
  threshold           = "75"

  dimensions = {
    AutoScalingGroupName = "${aws_autoscaling_group.app_asg.id}"
  }

  alarm_actions = ["${aws_autoscaling_policy.app_asg_scaleout_policy.arn}"]
}

resource "aws_autoscaling_policy" "app_asg_scalein_policy" {
  name                   = "${var.name}-${var.environment}-scalein-policy"
  scaling_adjustment     = -1
  adjustment_type        = "ChangeInCapacity"
  cooldown               = 900
  autoscaling_group_name = "${aws_autoscaling_group.app_asg.name}"
}

resource "aws_cloudwatch_metric_alarm" "asg_low_cpu" {
  alarm_name          = "${var.name}-${var.environment}-low-cpu"
  comparison_operator = "LessThanOrEqualToThreshold"
  evaluation_periods  = "15"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/EC2"
  period              = "60"
  statistic           = "Average"
  threshold           = "10"

  dimensions = {
    AutoScalingGroupName = "${aws_autoscaling_group.app_asg.id}"
  }

  alarm_actions = ["${aws_autoscaling_policy.app_asg_scalein_policy.arn}"]
}

module "asg_dns" {
  source     = "git::git+ssh://git@git-aws.internal.justin.tv/web/asg-r53-lambda.git//modules/notifications"
  topic_name = "${var.name}_${var.environment}_topic"
  asg_name   = "${aws_autoscaling_group.app_asg.name}"
  lambda_arn = "${data.terraform_remote_state.asg_lambda.asg_lambda_arn}"
}
