resource "aws_ecs_service" "app" {
  name = "${var.name}-${var.environment}"

  lifecycle {
    // Must ignore this so we can promote services to new task definitions
    ignore_changes = ["task_definition"]
  }

  task_definition = "${aws_ecs_task_definition.app.family}"
  desired_count   = "${var.desired_count}"
  cluster         = "${aws_ecs_cluster.service_cluster.id}"

  iam_role                           = "${data.terraform_remote_state.core.service_iam_role}"
  deployment_maximum_percent         = 200
  deployment_minimum_healthy_percent = 50

  load_balancer {
    target_group_arn = "${var.target_group_arn}"
    container_name   = "${data.template_file.app_task_defs.vars.name}"
    container_port   = "${var.container_port}"
  }

  placement_strategy {
    type  = "spread"
    field = "attribute:ecs.availability-zone"
  }

  placement_strategy {
    type  = "spread"
    field = "instanceId"
  }
}

resource "aws_ecs_cluster" service_cluster {
  name = "${var.cluster_name}"
}

# Placeholder task definition
data "template_file" "app_task_defs" {
  template = "${file("${path.module}/default_task_definition.json")}"

  vars {
    container_port = "${var.container_port}"
    name           = "${var.name}"
  }
}

resource "aws_ecs_task_definition" "app" {
  family                = "${var.name}-${var.environment}"
  task_role_arn         = "${var.task_role_arn}"
  container_definitions = "${data.template_file.app_task_defs.rendered}"
  network_mode          = "bridge"

  lifecycle {
    // Must ignore this so we can promote services to new task definitions
    ignore_changes = ["container_definitions"]
  }
}

data "terraform_remote_state" "core" {
  backend = "s3"

  config {
    bucket  = "${var.account}"
    key     = "tfstate/${var.team}/core/terraform/${var.environment}"
    region  = "${var.region}"
    profile = "${var.account}"
  }
}

# Output variables

output "service_name" {
  value = "${aws_ecs_service.app.name}"
}

output "cluster_name" {
  value = "${aws_ecs_cluster.service_cluster.name}"
}

output "task_family" {
  value = "${aws_ecs_task_definition.app.family}"
}
