# Task autoscaling
resource "aws_appautoscaling_target" "ecs_target" {
  max_capacity       = "${var.max_size}"
  min_capacity       = "${var.min_size}"
  resource_id        = "service/${var.cluster_name}/${var.service}"
  role_arn           = "${data.terraform_remote_state.core.task_autoscale_iam_role}"
  scalable_dimension = "ecs:service:DesiredCount"
  service_namespace  = "ecs"
}

resource "aws_appautoscaling_policy" "up" {
  name                    = "${var.cluster_name}-${var.service}-scale-up"
  service_namespace       = "ecs"
  resource_id             = "${aws_appautoscaling_target.ecs_target.resource_id}"
  scalable_dimension      = "ecs:service:DesiredCount"
  adjustment_type         = "ChangeInCapacity"
  cooldown                = 300
  metric_aggregation_type = "Maximum"

  step_adjustment {
    metric_interval_lower_bound = 0
    scaling_adjustment          = 1
  }

  depends_on = ["aws_appautoscaling_target.ecs_target"]
}

resource "aws_cloudwatch_metric_alarm" "cpu_high" {
  alarm_name          = "ecs-${var.cluster_name}-${var.service}-cpu-high"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  period              = "60"
  evaluation_periods  = "5"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/ECS"
  statistic           = "Maximum"
  threshold           = "85"

  dimensions {
    ClusterName = "${var.cluster_name}"
    ServiceName = "${var.service}"
  }

  alarm_actions = ["${aws_appautoscaling_policy.up.arn}"]
}

resource "aws_appautoscaling_policy" "down" {
  name                    = "${var.cluster_name}-${var.service}-scale-down"
  service_namespace       = "ecs"
  resource_id             = "${aws_appautoscaling_target.ecs_target.resource_id}"
  scalable_dimension      = "ecs:service:DesiredCount"
  adjustment_type         = "ChangeInCapacity"
  cooldown                = 300
  metric_aggregation_type = "Maximum"

  step_adjustment {
    metric_interval_lower_bound = 0
    scaling_adjustment          = -1
  }

  depends_on = ["aws_appautoscaling_target.ecs_target"]
}

resource "aws_cloudwatch_metric_alarm" "cpu_low" {
  alarm_name          = "ecs-${var.cluster_name}-${var.service}-cpu-low"
  comparison_operator = "LessThanOrEqualToThreshold"
  period              = "60"
  evaluation_periods  = "5"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/ECS"
  statistic           = "Maximum"
  threshold           = "25"

  dimensions {
    ClusterName = "${var.cluster_name}"
    ServiceName = "${var.service}"
  }

  alarm_actions = ["${aws_appautoscaling_policy.down.arn}"]
}

data "terraform_remote_state" "core" {
  backend = "s3"

  config {
    bucket  = "${var.account}"
    key     = "tfstate/${var.team}/core/terraform/${var.environment}"
    region  = "${var.region}"
    profile = "${var.account}"
  }
}
