#!/bin/bash
# This bash script creates a cloudformation template inside your account to setup Grafana->CloudWatch access.
# It then outputs a role ARN you should give to systems infrastructure to add to Grafana Data Sources.
set -euo pipefail

STACK_NAME=grafana-role
export AWS_DEFAULT_REGION=${AWS_DEFAULT_REGION-us-west-2}

function cli_works() {
	aws sts get-caller-identity &> /dev/null
}

# Returns non zero if the stack already exists
function stack_exists() {
	aws cloudformation describe-stacks --stack-name ${STACK_NAME} &> /dev/null
}

function aws_account_id() {
  aws sts get-caller-identity --query 'Account' --output text
}

if ! cli_works ; then
	echo "Unable to execute get-caller-identity"
	echo "Are you sure you setup your cli correctly?"
	echo "You may need to run mwinit or copy temp credentials from https://isengard.amazon.com"
	exit 1
fi

if ! stack_exists ; then
	echo "stack does not already exist.  Creating"
	echo "Creating stack"
	aws cloudformation create-stack --stack-name ${STACK_NAME} --template-body file://grafana_role.yaml --parameters file://log_groups.json --capabilities CAPABILITY_NAMED_IAM
	echo "Waiting for stack to finish creating"
	aws cloudformation wait stack-create-complete --stack-name ${STACK_NAME}
else
	echo "Stack already exists. Updating and pulling existing information.  If you get a 'No updates are to be performed' error, ignore it"
	aws cloudformation update-stack --stack-name ${STACK_NAME} --template-body file://grafana_role.yaml --parameters file://log_groups.json --capabilities CAPABILITY_NAMED_IAM 2> /tmp/err || true
	cat /tmp/err
	if ! grep -q  'No updates' /tmp/err; then
		aws cloudformation wait stack-update-complete --stack-name ${STACK_NAME}
	fi
fi

echo "Stack is setup"
echo "Create a ticket like https://jira.twitch.com/browse/IPSYS-911 with your account ID (it is probably $(aws_account_id))"
echo "Role ARN: arn:aws:iam::$(aws_account_id):role/grafana-cloudwatch-read-only"
