provider "aws" {
  region  = var.aws_region
  profile = var.aws_profile
}

variable "aws_profile" {
}

variable "aws_region" {
  default = "us-west-2"
}

variable "cloudwatch_log_group_arns" {
  default     = ["*"]
  description = "Permitted cloudwatch log group arns. Default is all."
}

variable "grafana_production_role_arns" {
  type = list(string)

  default = [
    # twitch-grafana-prod  (grafana.xarth.tv)
    "arn:aws:iam::109561199089:role/grafana-production",
    # twitch-grafana-dev   (grafana-dev.twitch.a2z.com)
    "arn:aws:iam::963768028156:role/grafana-development",
  ]

  description = "roles used by grafana clusters to fetch the metrics from cloudwatch. This role is managed by #availability team"
}

variable "cloudwatch_read_only_policy_arn" {
  default     = "arn:aws:iam::aws:policy/CloudWatchReadOnlyAccess"
  description = "Policy with permissions required to get cloudwatch metrics in grafana"
}

data "aws_iam_policy_document" "grafana-cloudwatch-arp" {
  statement {
    effect = "Allow"

    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type = "AWS"

      identifiers = var.grafana_production_role_arns
    }
  }
}

resource "aws_iam_role" "grafana-cloudwatch-read-only" {
  name               = "grafana-cloudwatch-read-only"
  assume_role_policy = data.aws_iam_policy_document.grafana-cloudwatch-arp.json
}

# https://grafana.com/docs/features/datasources/cloudwatch/#iam-policies
data "aws_iam_policy_document" "grafana-cloudwatch-access-policy" {
  statement {
    sid = "AllowReadingMetricsFromCloudWatch"

    actions = [
      "cloudwatch:DescribeAlarmsForMetric",
      "cloudwatch:DescribeAlarmHistory",
      "cloudwatch:DescribeAlarms",
      "cloudwatch:ListMetrics",
      "cloudwatch:GetMetricStatistics",
      "cloudwatch:GetMetricData",
    ]

    resources = [
      "*",
    ]
  }

  statement {
    sid = "AllowReadingTagsInstancesRegionsFromEC2"

    actions = [
      "ec2:DescribeTags",
      "ec2:DescribeInstances",
      "ec2:DescribeRegions",
    ]

    resources = [
      "*",
    ]
  }

  statement {
    sid = "AllowDescribingLogsFromCloudWatch"

    actions = [
      "logs:DescribeLogGroups",
      "logs:StopQuery",
      "logs:GetQueryResults",
    ]

    resources = ["*"]
  }

  statement {
    sid = "AllowReadingLogGroupsFromCloudWatch"

    actions = [
      "logs:GetLogGroupFields",
      "logs:StartQuery",
      "logs:GetLogEvents",
    ]

    resources = var.cloudwatch_log_group_arns
  }

  statement {
    sid = "AllowReadingResourcesForTags"

    actions = [
      "tag:GetResources",
    ]

    resources = [
      "*",
    ]
  }
}

resource "aws_iam_policy" "grafana_policy" {
  name   = "grafana-cloudwatch-access-policy"
  path   = "/"
  policy = data.aws_iam_policy_document.grafana-cloudwatch-access-policy.json
}

resource "aws_iam_role_policy_attachment" "grafana_policy_attach" {
  role       = aws_iam_role.grafana-cloudwatch-read-only.name
  policy_arn = aws_iam_policy.grafana_policy.arn
}

output "role_arn" {
  value = aws_iam_role.grafana-cloudwatch-read-only.arn
}

output "role_name" {
  value = aws_iam_role.grafana-cloudwatch-read-only.name
}

