#!/bin/bash

ISENGARD_CACHE_DIR="${ISENGARD_CACHE_DIR-"$HOME/.cache/isengard"}"

DIR="${BASH_SOURCE%/*}"
if [[ ! -d "$DIR" ]]; then DIR="$PWD"; fi

if [[ "${DEBUG-}" = "true" ]]; then
  set -x
fi

# Execute this make command in a new shell then return.  For example, to simulate `./make go test` you could call
# `make_run go test`
function make_run() {
  DIR="${BASH_SOURCE%/*}"
  if [[ ! -d "$DIR" ]]; then DIR="$PWD"; fi
  (
    cd ${DIR}/../..
    ./make.sh $@
  )
}

# Absolute path of $1, working in mac and linux
function abspath() {
  echo "$(cd "$(dirname "$1")"; pwd)/$(basename "$1")"
}

# Execute this make command replacing the current execution.  This is done via `exec`.  It does not return.  To simulate
# running `./make go test`, you can call `make_exec go test`
function make_exec() {
  DIR="${BASH_SOURCE%/*}"
  if [[ ! -d "$DIR" ]]; then DIR="$PWD"; fi
  cd ${DIR}/../..
  exec ./make.sh $@
}

# By default does nothing.  You can overwrite this function inside vars.sh to populate any environment variables your
# application needs when it runs locally
function populate_app_env() {
  return 0
}

function is_git_dirty {
  [[ $(git diff --shortstat 2> /dev/null | tail -n1) != "" ]]
}

# Expose env variables that allow us to run commands as an assumed role
function assume_role() {
  CREDS=$(aws sts assume-role $@ --query '[Credentials.AccessKeyId,Credentials.SecretAccessKey,Credentials.SessionToken]' --output text)
  if [ -z "${CREDS}" ]; then
    return 1
  fi
  read -ra CREDS_ARR <<< "$CREDS"
  export AWS_ACCESS_KEY_ID=${CREDS_ARR[0]}
  export AWS_SECRET_ACCESS_KEY=${CREDS_ARR[1]}
  export AWS_SESSION_TOKEN=${CREDS_ARR[2]}
}

function aws_account_id() {
  aws sts get-caller-identity --query 'Account' --output text
}

# Push Docker container to ECR
function ecr_push() {
  (
    INFRA=${1-}
    if [ -z ${INFRA} ]; then
      echo "First parameter required"
      exit 1
    fi
    DOCKER_TAG_PREFIX=${2-}
    ecr_push_core ${SERVICE}:${DOCKER_TAG_PREFIX}${GIT_COMMIT} ${SERVICE}-${INFRA}:${DOCKER_TAG_PREFIX}${GIT_COMMIT}
  )
}

# Returns true if the docker image exists locally
function docker_image_exists_locally() {
  IMAGE=${1-}
  if [ -z ${IMAGE} ]; then
    echo "First parameter required: Image to check for"
    exit 1
  fi
  docker image inspect ${IMAGE} > /dev/null
}

# Pull an image from ECR
function ecr_pull2() {
  (
    REMOTE_TAG=${1-}
    ACCOUNT=${2-}
    REMOTE_DOCKER_NAME=${ACCOUNT}.dkr.ecr.${AWS_DEFAULT_REGION}.amazonaws.com

    if aws ecr get-login-password help &>/dev/null; then
      aws ecr get-login-password --region ${AWS_DEFAULT_REGION} | docker login --username AWS --password-stdin ${REMOTE_DOCKER_NAME}
    else
      DOCKER_CMD=$(aws ecr get-login --no-include-email --registry-ids ${ACCOUNT})
      eval ${DOCKER_CMD}
    fi

    docker pull ${REMOTE_TAG}
  )
}

function ecr_push_core() {
  (
    ACCOUNT_ID=$(aws_account_id)
    LOCAL_TAG=${1-}
    REMOTE_TAG=${2-}
    REMOTE_DOCKER_NAME=${ACCOUNT_ID}.dkr.ecr.${AWS_DEFAULT_REGION}.amazonaws.com

    if aws ecr get-login-password help &>/dev/null; then
      aws ecr get-login-password --region ${AWS_DEFAULT_REGION} | docker login --username AWS --password-stdin ${REMOTE_DOCKER_NAME}
    else
      DOCKER_CMD=$(aws ecr get-login --no-include-email)
      eval ${DOCKER_CMD}
    fi

    docker tag ${LOCAL_TAG} ${REMOTE_DOCKER_NAME}/${REMOTE_TAG}
    docker push ${REMOTE_DOCKER_NAME}/${REMOTE_TAG}
  )
}

# Run protobuf, twirp, and protobuf linter
function default_protobuf_generation() {
  mkdir -p ./proto/${SERVICE}
  protoc --lint_out=. --proto_path=./proto    --twirp_out=paths=source_relative:./proto/${SERVICE} --go_out=paths=source_relative:./proto/${SERVICE} ${SERVICE}.proto
}

# Process a help flag for every subcommand
function process_help() {
  prefix=$1
  shift
  if [[ "${1-}" == "help" ]]; then
    echo "Prints each command and help"
    exit 0
  fi

  compgen -A function | grep $prefix | sort | while read -r line; do
    echo
    echo "  ${line:${#prefix}}"
    echo "    $($line help)"
  done
}


# Return true if this script is running inside jenkins
function in_jenkins() {
 [[ ! -z "${JENKINS_HOME-}" ]]
}

# Each builder has its own local copy of the ~/.cache directory that we do not throw away.  This allows us to
# do faster go build commands
function local_docker_cache_directory() {
  echo ${HOME}/.cache/builder_cache/$(sanitize_builder)
}

function sanitize_builder() {
  echo ${BUILDER} | sed -e 's/[^A-Za-z0-9._-]/_/g'
}

# Print out a cloudformation output value
function get_cloudformation_output() {
  local VAR=${1-}
  if [[ -z ${VAR} ]]; then
    exit 1
  fi
  local RES=$(aws cloudformation list-exports --query "Exports[?Name=='$1'].Value" --output text)
  if [[ -z ${RES} ]]; then
    exit 1
  fi
  echo ${RES}
}

# Print a cloudformation output variable, but caches it to disk so we don't have to go out each time
function cached_cloudformation_output() {
  local VAR=${1-}
  if [[ -z ${VAR} ]]; then
    exit 1
  fi
  local TD=${HOME}/.cache/cloudformation_output
  mkdir -p ${TD}
  CACHE_FILE=${TD}/$(sanitize_builder)${AWS_PROFILE-}${VAR}
  if [[ -f ${CACHE_FILE} ]]; then
    cat ${CACHE_FILE}
    exit 0
  fi
  VAL=$(get_cloudformation_output ${VAR})
  if [[ -z ${VAL} ]]; then
    exit 1
  fi
  echo ${VAL} > ${CACHE_FILE}
  echo ${VAL}
}

# This is the core to our __ expansion magic
function process_build() {
  prefix=$1
  shift
  T="$prefix${1-}"
  if declare -F "$T" >/dev/null ; then
    func="$prefix${1}"
    shift; # pop $1
    "$func" "$@"    # invoke our named function w/ all remaining arguments
  else
    echo "Undefined subcommand ${1-}"
    process_help $prefix
    exit 1
  fi
}

function kstatus() {
  [[ -e ~/.twinit/sshsocket ]]
}

# Assume the role applications run as inside an infrastructure
function assume_dev_role_from_infra() {
  SESSION_NAME="localrun-$(whoami)"
  assume_role --role-arn ${LOCAL_DEV_ROLE} --role-session-name ${SESSION_NAME}
}
