#!/bin/bash

export GO_VENDOR_MODE=${GO_VENDOR_MODE:-"true"}

function gomod_build_flag() {
  if [[ "${GO_VENDOR_MODE}" == "true" ]]; then
    echo "-mod=vendor"
  elif in_jenkins; then
    echo "-mod=readonly"
  fi
}

function root__go() {
  if [[ "${1-}" == "help" ]]; then
    echo "Runs go related commands"
    exit 0
  fi
  process_build go__ $@
}

function go__lint() {
  if [[ "${1-}" == "help" ]]; then
    echo "Runs gometalinter to lint your project"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run go lint $@
  fi
  golangci-lint run
}

function go_build_debug() {
  if [[ "${DEBUG-}" = "true" ]]; then
    echo "-v"
  fi
}

function go__build() {
  if [[ "${1-}" == "help" ]]; then
    echo "Build your go service locally"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run go build $@
  fi
  (
    goproxy_export_env
    go build $(go_build_debug) $(gomod_build_flag) ./cmd/${SERVICE}/
  )
}

function go__run() {
  # TODO: Verify the bastion tunnel is setup and fail fast if not
  if [[ "${1-}" == "help" ]]; then
    echo "Build then run the service locally"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    if [[ ! -z "${LOCAL_GO_RUN_DOCKER_ARGS-}" ]]; then
      # Note: must export this to be picked up by lower exec call
      export LOCAL_DOCKER_RUN_ARGS="${LOCAL_DOCKER_RUN_ARGS-} ${LOCAL_GO_RUN_DOCKER_ARGS}"
    fi
    make_exec builder run go run $@
  fi
  make_run go build
  (
    AWS_PROFILE="staging-deploy" assume_role --role-arn ${LOCAL_DEV_ROLE} --role-session-name "go_run"
    populate_app_env
    ./${SERVICE} $@
  )
}

function go__module_verify() {
  if [[ "${1-}" == "help" ]]; then
    echo "Verify go.mod and go.sum files"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run go module_verify $@
  fi
  (
    goproxy_export_env
    go mod download
    go mod verify
  )
}
function go__integration_test() {
  if [[ "${1-}" == "help" ]]; then
    echo "Run integration tests locally"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run go integration_test $@
  fi
  (
    goproxy_export_env
    AWS_PROFILE="staging-deploy" assume_role --role-arn ${LOCAL_DEV_ROLE} --role-session-name "go_run"
    populate_app_env
    go test -count=1 $(gomod_build_flag) -v -timeout ${INTEGRATION_TEST_TIMEOUT} -race -run="TestIntegration" -tags=integration ./...
  )
}

function go__test() {
  if [[ "${1-}" == "help" ]]; then
    echo "Run the service's Go unit tests"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run go test $@
  fi
  (
    goproxy_export_env
    go test -count=1 $(gomod_build_flag) -race -cover -coverprofile=coverage.txt -covermode=atomic -timeout 30s ./...
  )
}

function go__format() {
  if [[ "${1-}" == "help" ]]; then
    echo "Reformat all go files"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run go format $@
  fi
  find . -iname '*.go' -not -path '*/vendor/*' -not -path '*/_tools/*' -print0 | xargs -0 gofmt -s -w
  find . -iname '*.go' -not -path '*/vendor/*' -not -path '*/_tools/*' -print0 | xargs -0 env GO111MODULE=off goimports -w
}
