#!/bin/bash

function root__infra() {
  if [[ "${1-}" == "help" ]]; then
    echo "Runs infrastructure related commands"
    exit 0
  fi
  if [[ "${USE_BUILDER}" == "true" ]]; then
    make_exec builder run infra $@
  fi
  process_build infra__ $@
}

function print_valid_infra() {
  find ./cloudformation/infra -type f -name '*.json' | sed "s/.*\///" | sed "s#\..*##"
}

function print_valid_infra_deploy() {
  find ./cloudformation/deploy -type f -name '*.json' | sed "s/.*\///" | sed "s#\..*##"
}

function infra__manage() {
  if [[ "${1-}" == "help" ]]; then
    echo "Update pipeline infrastructure"
    exit 0
  fi
  # If jenkins, setup for deploy
  # Otherwise, setup for local
  # I could do it here... setup the file correctly for jenkins
  cfmanage $@
}

function require-argument() {
  [[ -n "$2" ]] || (echo "Missing required argument $1"; exit 1)
}


function infra__docker_push() {
  if [[ "${1-}" == "help" ]]; then
    echo "Push a docker container to an infrastructure"
    exit 0
  fi
  INFRA="${1-}"
  if [[ -z ${INFRA} ]]; then
    echo "Pick an infra.  Valid infra to docker push: "
    echo
    print_valid_infra
    exit 1
  fi
  PUSH_INFRA=${INFRA}
  if [[ ${PUSH_INFRA} == "personal" ]]; then
    PUSH_INFRA=${INFRA}${RUNNER}
  fi
  ecr_push ${PUSH_INFRA} app-
}

function infra__docker_integration_push() {
  if [[ "${1-}" == "help" ]]; then
    echo "Push an integration docker container to an infrastructure"
    exit 0
  fi
  INFRA="${1-}"
  if [[ -z ${INFRA} ]]; then
    echo "Pick an infra.  Valid infra to docker push: "
    echo
    print_valid_infra
    exit 1
  fi
  # TODO: Show valid infra to update
  PUSH_INFRA=${INFRA}
  if [[ ${PUSH_INFRA} == "personal" ]]; then
    PUSH_INFRA=${INFRA}${RUNNER}
  fi
  ecr_push ${PUSH_INFRA} integration-
}

function infra__output() {
  if [[ "${1-}" == "help" ]]; then
    echo "Print a CloudFormation output inside the infrastructure"
    exit 0
  fi
  INFRA="${1-}"
  if [ -z ${INFRA} ]; then
    echo "Pick an infra.  Valid infra to print from"
    echo
    print_valid_infra
    exit 1
  fi
  shift
  infra_assume_infra_role ${INFRA}
  VAL="${1-}"
  if [[ -z ${VAL} ]]; then
    echo "Pick a value to output"
    echo
    aws cloudformation list-exports --query "Exports[*].[Name]" --output text
    exit 1
  fi
  shift
  get_cloudformation_output ${VAL}
}

function infra__integration_test_ecs_run() {
  if [[ "${1-}" == "help" ]]; then
    echo "Run the integration tests inside ECS remotely"
    exit 0
  fi
  populate_app_env
  local RUNPARAMS=""
  if [[ "${DEBUG-}" = "true" ]]; then
    RUNPARAMS="-verbosity=2"
  fi
  ecsrun -prefix cmd/${SERVICE}/integration ${RUNPARAMS}
}

function infra__guess_info() {
  if [[ "${1-}" == "help" ]]; then
    echo "Guess information about your dev and production account"
    exit 0
  fi

  VPC_ID=$(guess_vpc_id)
  echo "AWS region"
  echo "  ${AWS_REGION}"
  echo "Account ID"
  echo "  $(guess_aws_account_id)"
  echo "VPC id"
  echo "  ${VPC_ID}"
  echo "Security group"
  echo "  $(guess_security_group_id)"
  echo "Private Subnets"
  echo "  $(guess_private_subnet_id ${VPC_ID})"

}

# Discovery functions from https://git.xarth.tv/twitch/basic_vpc

function discover_basic_private_subnets() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='AllPrivateSubnets'].OutputValue" --output text
}

function discover_basic_vpc() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='VPC'].OutputValue" --output text
}

function discover_basic_security_group() {
    aws cloudformation describe-stacks --stack-name vpc --query "Stacks[0].Outputs[?OutputKey=='SecurityGroup'].OutputValue" --output text
}

# Find the current account ID in AWS_PROFILE
function guess_aws_account_id() {
  aws sts get-caller-identity --query 'Account' --output text
}

function guess_vpc_id() {
  BASIC_ID=$(discover_basic_vpc)
  if [[ ! -z ${BASIC_ID} ]]; then
    echo ${BASIC_ID}
    return 0
  fi
  aws ec2 describe-vpcs --filters "Name=isDefault,Values=false" --query 'Vpcs[].VpcId' --output text
}

function guess_security_group_id() {
  BASIC_ID=$(discover_basic_security_group)
  if [[ ! -z ${BASIC_ID} ]]; then
    echo ${BASIC_ID}
    return 0
  fi
  aws ec2 describe-security-groups --query "SecurityGroups[?GroupName=='twitch_subnets'].GroupId" --output text
}

function guess_private_subnet_id() {
  VPC_ID=$1
  BASIC_ID=$(discover_basic_private_subnets)
  if [[ ! -z ${BASIC_ID} ]]; then
    echo ${BASIC_ID}
    return 0
  fi
  aws ec2 describe-subnets --query "Subnets[? ! MapPublicIpOnLaunch] | [?VpcId =='$VPC_ID'].SubnetId | join(',', @)" --output text
}
