#!/bin/bash
# This bash script creates a cloudformation template inside your account to setup the Jenkins user.  It then
# outputs a credentials file you can upload to jenkins so your jenkins jobs can communicate with your AWS account.
set -euo pipefail

STACK_NAME=jenkins-limited-user
if [ -z "${AWS_PROFILE-}" ]; then
	echo "Unable to find AWS_PROFILE env variable.  Set it to the account you want the jenkins user created inside"
	echo "This assumes the aws-cli is already configured to communicate with that account"
	exit 1
fi

function cli_works() {
	aws sts get-caller-identity &> /dev/null
}

# Returns non zero if the stack already exists
function stack_exists() {
	aws cloudformation describe-stacks --stack-name ${STACK_NAME} &> /dev/null
}

# Loads the secret key from the previously created stack
function secret_key() {
	aws cloudformation describe-stacks --stack-name ${STACK_NAME} --query 'Stacks[0].Outputs[0].OutputValue' --output text
}

# Load the access key from the previously created stack
function access_key() {
	aws cloudformation describe-stack-resources --stack-name ${STACK_NAME} --query "StackResources[?LogicalResourceId=='LimitedJenkinsAccessKey'].PhysicalResourceId" --output text
}

if ! cli_works ; then
	echo "Unable to execute get-caller-identity for ${AWS_PROFILE}"
	echo "Are you sure you setup your cli correctly?"
	exit 1
fi

if ! stack_exists ; then
	echo "stack does not already exist.  Creating"
	echo "Creating stack"
	aws cloudformation create-stack --stack-name ${STACK_NAME} --template-body file://cloudformation/jenkins/jenkins.yaml --capabilities CAPABILITY_NAMED_IAM
	echo "Waiting for stack to finish creating"
	aws cloudformation wait stack-create-complete --stack-name ${STACK_NAME}
else
	echo "Stack already exists. Updating and pulling existing information.  If you get a 'No updates are to be performed' error, ignore it"
	aws cloudformation update-stack --stack-name ${STACK_NAME} --template-body file://cloudformation/jenkins/jenkins.yaml --capabilities CAPABILITY_NAMED_IAM 2> /tmp/err || true
	cat /tmp/err
	if ! grep -q  'No updates' /tmp/err; then
		aws cloudformation wait stack-update-complete --stack-name ${STACK_NAME}
	fi
fi

secret=$(secret_key)
access=$(access_key)

FILE_CONTENTS=$(echo "
[${AWS_PROFILE}]
aws_access_key_id = $access
aws_secret_access_key = $secret
")

#echo "$FILE_CONTENTS"  Silence the secret keys
echo "$FILE_CONTENTS" > "${AWS_PROFILE}"
echo 
echo "Done!"
echo
echo "There is now a file named ${AWS_PROFILE}.  Upload this file to jenkins as a credential from https://jenkins.internal.justin.tv/credentials/store/system/domain/discovery/"
echo "Upload it as type 'Secret file' and with id ${AWS_PROFILE}"
echo "Then you can use this credential in your Jenkinsfile like this...
    stage('Production docker push') {
      when {
        anyOf {
          branch 'master';
        }
     }
      steps {
        withCredentials([file(credentialsId: '${AWS_PROFILE}', variable: 'AWS_SHARED_CREDENTIALS_FILE')]) {
          sh 'AWS_PROFILE=${AWS_PROFILE} aws s3 blah'
        }
      }
    }
"
