#!/bin/bash
# Uses cloudformation to create an LDAP private link
set -euo pipefail

STACK_NAME=ldap-private-link
if [[ -z "${ACCOUNT_NAME}" ]]; then
	echo "Unable to find ACCOUNT_NAME env variable.  Set it to the account name systems setup your VPC with.  If you"
	echo "do not, we won't be able to auto detect your security group information"
	exit 1
fi

function cli_works() {
	aws sts get-caller-identity &> /dev/null
}

# Returns non zero if the stack already exists
function stack_exists() {
	aws cloudformation describe-stacks --stack-name ${STACK_NAME} &> /dev/null
}

function guess_vpc_ids() {
  if [[ ! -z "${VPC_ID-}" ]]; then
    echo ${VPC_ID}
    exit 0
  fi
	aws ec2 describe-vpcs --filters "Name=tag:Name,Values=${ACCOUNT_NAME}" --query 'Vpcs[].VpcId' --output text
}

function guess_security_group_id() {
  if [[ ! -z "${SECURITY_GROUP_ID-}" ]]; then
    echo ${SECURITY_GROUP_ID}
    exit 0
  fi
  aws ec2 describe-security-groups --query "SecurityGroups[?GroupName=='twitch_subnets'].GroupId" --output text
}

function guess_all_private_subnet_ids() {
  if [ ! -z "${SUBNET_ID-}" ]; then
    echo ${SUBNET_ID}
    exit 0
  fi
  VPC_ID=$1
  aws ec2 describe-subnets --query "join(',', Subnets[? ! MapPublicIpOnLaunch] | [?VpcId =='$VPC_ID'].SubnetId)" --output json
}

if ! cli_works ; then
	echo "Unable to execute get-caller-identity for ${ACCOUNT_NAME}"
	echo "Are you sure you setup your cli correctly?"
	exit 1
fi

aws sts get-caller-identity

SCRIPT_VPC_ID=$(guess_vpc_ids)
SCRIPT_SG_ID=$(guess_security_group_id)
SCRIPT_SUBNET_ID=$(guess_all_private_subnet_ids ${SCRIPT_VPC_ID})

if [[ -z "$SCRIPT_VPC_ID" ]]; then
  echo "Unable to guess a VPC_ID for this script.  Pass one with 'env VPC_ID=XYZ'"
  exit 1
fi

if [[ -z "$SCRIPT_SG_ID" ]]; then
  echo "Unable to guess a security group ID for this script.  Pass one with 'env SECURITY_GROUP_ID=XYZ'"
  exit 1
fi

if [[ -z "$SCRIPT_SUBNET_ID" ]]; then
  echo "Unable to guess a subnet ID for this script.  Pass one with 'env SUBNET_ID=XYZ'"
  exit 1
fi

PARAMS="ParameterKey=Subnets,ParameterValue=${SCRIPT_SUBNET_ID} ParameterKey=SecurityGroups,ParameterValue=${SCRIPT_SG_ID} ParameterKey=VpcID,ParameterValue=${SCRIPT_VPC_ID}"

echo "Using params $PARAMS"

if stack_exists ; then
	echo "The stack already exists.  Visit the cloudformation UI and ask #systems to accept your LDAP link.  Will update the stack: just in case.  Ignore any errors about 'no updates to change'"
	aws cloudformation update-stack --stack-name ${STACK_NAME} --template-body file://ldap_plink.yaml --parameters ${PARAMS} --capabilities CAPABILITY_NAMED_IAM
	exit 1
fi

## Large timeout since we need to wait a long time for them to accept LDAP link
aws cloudformation create-stack --timeout-in-minutes  10080 --stack-name ${STACK_NAME} --template-body file://ldap_plink.yaml --parameters ${PARAMS} --capabilities CAPABILITY_NAMED_IAM

echo "Done!"
echo
echo "Please ask in #systems for someone to accept your LDAP private link"
echo "In 7 days your cloudformation stack will roll back if the private link is not approved"
