#!/bin/bash
# Verify DNS of created LDAP link
set -e

STACK_NAME=ldap-private-link
if [[ -z "${ACCOUNT_NAME}" ]]; then
	echo "Unable to find ACCOUNT_NAME env variable.  Set it to the account name systems setup your VPC with.  If you"
	echo "do not, we won't be able to auto detect your security group information"
	exit 1
fi

function cli_works() {
	aws sts get-caller-identity &> /dev/null
}

# Returns non zero if the stack already exists
function stack_exists() {
	aws cloudformation describe-stacks --stack-name $STACK_NAME &> /dev/null
}

function ldap_vpce() {
  aws cloudformation describe-stacks --stack-name ldap-private-link --query 'Stacks[0].Outputs[0].OutputValue' --output text
}

if ! cli_works ; then
	echo "Unable to execute get-caller-identity for ${ACCOUNT_NAME}"
	echo "Are you sure you setup your cli correctly?"
	exit 1
fi

VPC=$(ldap_vpce)
echo "We found the created VPC $VPC"
echo "Finding DNS name:"
DNS_NAME=$(aws ec2 describe-vpc-endpoints --vpc-endpoint-ids "$VPC" | jq -r .VpcEndpoints[0].DnsEntries[0].DnsName)
echo "Found DNS name:
 $DNS_NAME"
echo "
Execute this command on a remote host to verify LDAP connectivity (optional):
  openssl s_client -connect ${DNS_NAME}:636 | openssl x509 -noout -text"

echo "
Now setup a DNS entry.
1. Visit https://dashboard.internal.justin.tv/dns/?entry=ldap-${ACCOUNT_NAME}.internal.justin.tv
2. Click 'CNAME Record' and 'Create'
3. Add an internal CNAME record for ${DNS_NAME}
4. Click 'Save all'"

echo "
Once this is done, you can verify LDAP connectivity by executing on a host:
  ldapsearch -x -H ldaps://ldap-${ACCOUNT_NAME}.internal.justin.tv   -b 'dc=justin,dc=tv' 'uid=ccmolik'
"
