# Creates an ALB and exports the ALB target group

## User warning
## You may need to prepare your terraform twice because of a dependency between the ALB listener and ecs group
# See https://github.com/hashicorp/terraform/issues/12634 for when this is fixed

resource "aws_alb_target_group" "app" {
  name                 = "${var.team}-${var.service}-${var.environment}"
  deregistration_delay = 30

  # Use a dummy low port because we want applications to give us their port
  port     = "1"
  protocol = "HTTP"
  vpc_id   = var.vpc_id

  health_check {
    path     = "/debug/running"
    protocol = "HTTP"
    matcher  = "200"
  }

  tags = {
    Service     = var.service
    Environment = var.environment
    Owner       = var.owner
    Team        = var.team
  }
}

resource "aws_alb_listener" "app" {
  load_balancer_arn = aws_alb.app.arn
  port              = "80"
  protocol          = "HTTP"

  default_action {
    target_group_arn = aws_alb_target_group.app.arn
    type             = "forward"
  }
}

resource "aws_alb" "app" {
  name                       = "${var.team}-${var.service}-${var.environment}"
  security_groups            = [var.security_groups]
  subnets                    = split(",", var.subnets)
  enable_deletion_protection = true
  internal                   = true

  access_logs {
    bucket  = var.access_log_bucket
    prefix  = "elb-logs/${var.team}-${var.service}-${var.environment}"
    enabled = true
  }

  tags = {
    Name        = var.service
    Service     = var.service
    Environment = var.environment
    Owner       = var.owner
    Team        = var.team
  }
}

output "target_group_arn" {
  value = aws_alb_target_group.app.arn
}

output alb_arn {
  value = aws_alb.app.arn
}

output "dns_name" {
  value = aws_alb.app.dns_name
}

output "zone_id" {
  value = aws_alb.app.zone_id
}

module "consulconf" {
  source      = "../consulconf"
  team        = var.team
  service     = var.service
  environment = var.environment
  key         = "alb_dns"
  value       = aws_alb.app.dns_name
}
