variable "table_name" {
  description = "The name of the dynamodb table to monitor"
}

variable "alarm_threshold_percent" {
  description = "What % of read or write before we trigger alarm"
  default     = 70
}

variable "alarm_period" {
  description = "Seconds to wait till it triggers"
  default     = 60
}

variable "evaluation_periods" {
  default = 5
}

variable "alerts_sns_arn" {
  description = "SNS ARN to send alerts"
}

variable "read_capacity" {
  description = "The read capacity of the input table"
}

variable "write_capacity" {
  description = "The write capacity of the input table"
}

resource "aws_cloudwatch_metric_alarm" "read_capacity_alarm" {
  alarm_name        = "${var.table_name}-read-limit"
  alarm_description = "Alerts when ConsumedReadCapacityUnits for ${var.table_name} exceeds ${var.alarm_threshold_percent}% for ${var.alarm_period} seconds"
  namespace         = "AWS/DynamoDB"

  dimensions = {
    TableName = var.table_name
  }

  metric_name         = "ConsumedReadCapacityUnits"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  statistic           = "Sum"
  period              = var.alarm_period
  evaluation_periods  = var.evaluation_periods
  threshold           = var.read_capacity * var.alarm_threshold_percent * var.alarm_period / 100
  alarm_actions       = [var.alerts_sns_arn]
  ok_actions          = [var.alerts_sns_arn]
  actions_enabled     = true
}

resource "aws_cloudwatch_metric_alarm" "write_capacity_alarm" {
  alarm_name        = "${var.table_name}-write-limit"
  alarm_description = "Alerts when ConsumedWriteCapacityUnits for ${var.table_name} exceeds ${var.alarm_threshold_percent}% for ${var.alarm_period} seconds"
  namespace         = "AWS/DynamoDB"

  dimensions = {
    TableName = var.table_name
  }

  metric_name         = "ConsumedWriteCapacityUnits"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  statistic           = "Sum"
  period              = var.alarm_period
  evaluation_periods  = var.evaluation_periods
  threshold           = var.write_capacity * var.alarm_threshold_percent * var.alarm_period / 100
  alarm_actions       = [var.alerts_sns_arn]
  ok_actions          = [var.alerts_sns_arn]
  actions_enabled     = true
}

