# Primary application ASG and ELB

locals {
  ami       = coalesce(var.ami, data.aws_ami.ecs_ami.id)
  ldap_host = coalesce(var.custom_ldap_host, local.internal_ldap_host)
}

data "aws_ami" "ecs_ami" {
  most_recent = true

  owners = ["amazon"]

  // Note: If the name pattern on https://docs.aws.amazon.com/AmazonECS/latest/developerguide/al2ami.html
  //       is ever updated, we need to change this filter
  filter {
    name   = "name"
    values = ["amzn2-ami-ecs-hvm-*-x86_64-ebs"]
  }
}

resource "aws_launch_configuration" "app_conf" {
  name_prefix     = "${var.team}-${var.environment}-${var.name}"
  image_id        = local.ami
  instance_type   = var.instance_type
  security_groups = split(",", var.security_groups)
  user_data       = data.template_file.app_user_data.rendered

  key_name             = var.host_key_name
  iam_instance_profile = var.container_instance_profile_id

  root_block_device {
    volume_size = var.root_volume_size
  }

  lifecycle {
    create_before_destroy = true
  }
}

data "template_file" "app_user_data" {
  template = file("${path.module}/asg_userdata.template")

  vars = {
    ecs_cluster_name           = var.cluster_name
    ldap_host                  = local.ldap_host
    ldap_group                 = var.ldap_group
    environment                = var.environment
    ssm_agent_loggroup         = var.ssm_agent_loggroup
    ecs_agent_loggroup         = var.ecs_agent_loggroup
    ecs_init_loggroup          = var.ecs_init_loggroup
    attribute_json             = jsonencode(var.instance_attributes)
    task_cleanup_wait_duration = var.task_cleanup_wait_duration
  }
}

resource "aws_autoscaling_group" "app_asg" {
  name                      = "${var.team}-${var.environment}-${var.name}"
  vpc_zone_identifier       = split(",", var.vpc_zone_identifier)
  max_size                  = var.max_size
  min_size                  = var.min_size
  desired_capacity          = var.desired_capacity
  health_check_grace_period = 600
  health_check_type         = "EC2"
  launch_configuration      = aws_launch_configuration.app_conf.name
  enabled_metrics = [
    "GroupDesiredCapacity",
    "GroupInServiceInstances",
    "GroupMaxSize",
    "GroupMinSize",
    "GroupPendingInstances",
    "GroupStandbyInstances",
    "GroupTerminatingInstances",
    "GroupTotalInstances",
  ]

  tag {
    key                 = "Name"
    value               = var.name
    propagate_at_launch = true
  }

  tag {
    key                 = "Environment"
    value               = var.environment
    propagate_at_launch = true
  }

  tag {
    key                 = "Owner"
    value               = var.owner
    propagate_at_launch = true
  }

  tag {
    key                 = "Team"
    value               = var.team
    propagate_at_launch = true
  }

  // If we don't ignore this, then each rerun of terraform resets the desired capacity if it ever auto scaled
  lifecycle {
    ignore_changes = [desired_capacity]
  }
}

resource "aws_autoscaling_policy" "app_asg_scaleout_policy" {
  name                   = "${var.team}-${var.environment}-${var.name}-scaleout-policy"
  scaling_adjustment     = var.scale_out_increment
  adjustment_type        = "ChangeInCapacity"
  cooldown               = var.scale_out_cooldown
  autoscaling_group_name = aws_autoscaling_group.app_asg.name
}

resource "aws_cloudwatch_metric_alarm" "asg_high_cpu" {
  alarm_name          = "${var.team}-${var.environment}-${var.name}-high-cpu"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = var.scale_out_evaluation_periods
  metric_name         = "CPUUtilization"
  namespace           = "AWS/EC2"
  period              = "60"
  statistic           = "Average"
  threshold           = var.scale_out_cpu_percent_threshold

  dimensions = {
    AutoScalingGroupName = aws_autoscaling_group.app_asg.id
  }

  alarm_actions = [aws_autoscaling_policy.app_asg_scaleout_policy.arn]
}

resource "aws_autoscaling_policy" "app_asg_scalein_policy" {
  name                   = "${var.team}-${var.environment}-${var.name}-scalein-policy"
  scaling_adjustment     = var.scale_in_increment
  adjustment_type        = "ChangeInCapacity"
  cooldown               = var.scale_in_cooldown
  autoscaling_group_name = aws_autoscaling_group.app_asg.name
}

resource "aws_cloudwatch_metric_alarm" "asg_low_cpu" {
  alarm_name          = "${var.team}-${var.environment}-${var.name}-low-cpu"
  comparison_operator = "LessThanOrEqualToThreshold"
  evaluation_periods  = var.scale_in_evaluation_periods
  metric_name         = "CPUUtilization"
  namespace           = "AWS/EC2"
  period              = "60"
  statistic           = "Average"
  threshold           = var.scale_in_cpu_percent_threshold

  dimensions = {
    AutoScalingGroupName = aws_autoscaling_group.app_asg.id
  }

  alarm_actions = [aws_autoscaling_policy.app_asg_scalein_policy.arn]
}

