from aws_cdk import (
    core,
    aws_imagebuilder as cdk_ib,
    aws_ssm as cdk_ssm,
)
from ami_imagebuilder_cdk.image_recipe_construct import ImageRecipeConstruct
from ami_imagebuilder_cdk.scaffolding_construct import ScaffoldingConstruct
from ami_imagebuilder_cdk.distribution_config_construct import DistributionConfigConstruct
from ami_imagebuilder_cdk.infrastructure_config_construct import InfrastructureConfigConstruct
from ami_imagebuilder_cdk.base_build_components_construct import BaseBuildComponentsConstruct
from ami_imagebuilder_cdk.package_build_components_construct import PackageBuildComponentsConstruct
from ami_imagebuilder_cdk.outpost_build_components_construct import OutpostBuildComponentsConstruct

from ami_imagebuilder_cdk.ami import AMI
from ami_imagebuilder_cdk.proxy import NO_PROXY


class AmiBuildPipelineStack(core.Stack):
    def __init__(self, scope: core.Construct, id: str,
            ami: AMI,
            build_component_arns=None,
            **kwargs) -> None:
        """
        :param `AMI` ami: AMI configuration for the Build Pipeline
        """
        super().__init__(scope, id, **kwargs)

        # bring along the scaffolding
        scaffold = ScaffoldingConstruct(self, "Scaffold")

        # use some base build components
        build_components = BaseBuildComponentsConstruct(self, "BaseComponents").component_arns
        if len(ami.packages) > 0:
            build_components.extend(PackageBuildComponentsConstruct(self, "MoreComponents", ami.packages, ami.version).component_arns)

        if ami.outpost:
            build_components.extend(OutpostBuildComponentsConstruct(self, "OutpostComponents").component_arns)

        if build_component_arns is not None and len(build_component_arns) > 0:
            build_components.extend(build_component_arns)

        # image recipe needs:
        #   build_component_arns
        #   version of recipe (which results in AMI version)
        #   parent_image_arn
        #   block device mappings (hardcoded for now)
        recipe = ImageRecipeConstruct(self, "ImageRecipe",
                                      build_component_arns=build_components,
                                      ami=ami,
                                      )

        # infra config needs:
        #   instance types for building
        #   instance role for building
        #   bucket used for logging
        #   optional VPC specific subnet ids or security group ids
        infra_config = InfrastructureConfigConstruct(self, "InfraConfig",
                                                     instance_profile=scaffold.instance_profile,
                                                     logging_bucket=scaffold.logging_bucket,
                                                     )

        # distribution config describes what regions and which accounts to distribute
        # final AMI to. needs:
        #   ami_name
        #   ami_version
        #   distribute_regions
        #   distribute_accounts
        distribution_config = DistributionConfigConstruct(self, "DistribConfig",
                                                          ami_name=ami.name,
                                                          ami_version=ami.version,
                                                          distribute_regions=ami.distribute_regions,
                                                          distribute_accounts=ami.distribute_accounts,
                                                          )

        # configure pipeline with all the pieces, needs:
        #   image recipe
        #   distribution config
        #   infra config
        cdk_ib.CfnImagePipeline(self, "Pipeline",
                                name=core.Stack.of(self).stack_name,
                                image_recipe_arn=recipe.attr_arn,
                                description=ami.description,
                                infrastructure_configuration_arn=infra_config.attr_arn,
                                distribution_configuration_arn=distribution_config.attr_arn
                                )
