import yaml
import os
import re
from aws_cdk import (
    core,
    aws_imagebuilder as cdk_ib,
)


class BaseBuildComponentsConstruct(core.Construct):
    """Build Components to be resused across ImageBuilder stacks

    This reads in all included build_components from the git repo.

    These are meant to be the base building blocks for the AMI.
    To be maintainted by the VIDCS team.
    """

    def __init__(self, scope: core.Construct, id: str, **kwargs) -> None:
        super().__init__(scope, id, **kwargs)

        self._stack_name = core.Stack.of(self).stack_name

        bcs = []
        dirname = os.path.dirname(__file__)
        base_dir = os.path.join(dirname, 'build_components')
        # for each first level directory in base_dir
        for bc in sorted(next(os.walk(base_dir))[1]):
            version_file = os.path.join(base_dir, bc, 'VERSION')
            component_file = os.path.join(base_dir, bc, '{}.yaml'.format(bc))
            with open(version_file) as f:
                component_version = f.read().strip()
            with open(component_file) as f:
                data = f.read()
            component_name = re.sub(r'[0-9_]', '', bc)
            # XXX: cannot modify build component after creation
            # uniqueness is derived from Name, Version instead of ID
            # meaning that the assigned name should be stack depenendent
            build_component = cdk_ib.CfnComponent(self,
                                                  "BuildCmpt{}{}".format(component_name.title(), component_version),
                                                  name='{}-{}'.format(self._stack_name, component_name),
                                                  description=yaml.load(data, Loader=yaml.FullLoader).get('description'),
                                                  platform="Linux",
                                                  version=component_version,
                                                  data=data,
                                                  )
            bcs.append(build_component)

        self._components = bcs

    def _format_arn(self, name):
        return core.Stack.of(self).format_arn(
            partition='aws',
            service='imagebuilder',
            account='aws',
            resource='component',
            resource_name=name,
        )

    def _default_build_component_arns(self):
        """Returns ARNs of default base Amazon provided components for easy inclusion"""
        amzn_components = [
            'amazon-cloudwatch-agent-linux/x.x.x',
            'aws-codedeploy-agent-linux/x.x.x',
            'aws-cli-version-2-linux/x.x.x',
        ]
        return [self._format_arn(name) for name in amzn_components]

    @property
    def component_arns(self):
        return [c.attr_arn for c in self._components] + self._default_build_component_arns()
