from aws_cdk import core
from aws_cdk.aws_imagebuilder import CfnDistributionConfiguration as DistribConfig


class DistributionConfigConstruct(core.Construct):

    def __init__(self, scope: core.Construct, id: str, *,
                 ami_name, ami_version, ami_tags=None, description=None,
                 distribute_accounts=list(), distribute_regions=list(),
                 **kwargs) -> None:
        """
        Defines a distribution configuration

        :param str ami_name: Name of AMI
        :param str ami_version: Version of AMI
        :param dict ami_tags: Tags to tag onto distributed AMI
        :param str description: Description of AMI
        :param list distribute_accounts: Account IDs to distribute to
        :param list distribute_regions: Regions to distribute to
        """
        super().__init__(scope, id, **kwargs)

        if ami_tags is None:
            ami_tags = {"Name": ami_name, "Version": ami_version}

        if len(distribute_regions) == 0:
            distribute_regions = [core.Aws.REGION]

        distribute_accounts = set(distribute_accounts)
        distribute_regions = set(distribute_regions)

        # region specific distribution template, specifies which accounts to distribute to
        # note that buildDate *must* be present in the name
        distro_template = {
            "Name": "{} {}-{{{{imagebuilder:buildVersion}}}} {{{{imagebuilder:buildDate}}}}".format(ami_name, ami_version),
            "AmiTags": ami_tags,
            "Description": description,
        }

        if len(distribute_accounts) > 0:
            distro_template["LaunchPermissionConfiguration"] = {
                "UserIds": [str(account) for account in distribute_accounts],
            }

        # helpful vanity description
        conf_description = 'Distribute {} to {} and tag with ({})'.format(
            ami_name,
            ', '.join(distribute_regions),
            ', '.join(ami_tags.keys()),
        )

        self._distribution_config = DistribConfig(self, '1',
                                                  name=scope.stack_name,
                                                  description=conf_description,
                                                  distributions=[DistribConfig.DistributionProperty(
                                                      region=r,
                                                      ami_distribution_configuration=distro_template,
                                                  )
                                                      for r in distribute_regions]
                                                  )

    @property
    def attr_arn(self):
        return self._distribution_config.attr_arn
