from aws_cdk import (
    core,
    aws_imagebuilder as cdk_ib
)


class ImageRecipeConstruct(core.Construct):
    """Configure the Image recipe associated with the pipeline
    """
    def __init__(self, scope: core.Construct, id: str, *,
                 ami, build_component_arns,
                 description=None, **kwargs) -> None:
        """
        :param str ami: AMI config
        :param list build_component_arns: List of component arns to attach to the ImageRecipe
        :param str description: Description of ImageRecipe

        Not Implemented yet: os, os_release
        """
        super().__init__(scope, id, **kwargs)

        parent_image_arn = core.Stack.of(self).format_arn(
            partition='aws',
            service='imagebuilder',
            account='aws',
            resource='image',
            resource_name='{}/{}'.format(ami.parent_image, ami.parent_image_version),
        )

        ebs = cdk_ib.CfnImageRecipe.EbsInstanceBlockDeviceSpecificationProperty(**ami._get_ebs_spec())
        # XXX: you cannot modify an ImageRecipe after creation
        # uniqueness is derived from Name, Version instead of ID
        self._recipe = cdk_ib.CfnImageRecipe(self, '1',
                                             name=scope.stack_name,
                                             version=ami.version,
                                             description=description,
                                             components=[{"componentArn": arn} for arn in build_component_arns],
                                             parent_image=parent_image_arn,
                                             block_device_mappings=[cdk_ib.CfnImageRecipe.InstanceBlockDeviceMappingProperty(
                                                 device_name="/dev/sda1",
                                                 ebs=ebs,
                                             )]
                                             )

    @property
    def attr_arn(self):
        return self._recipe.attr_arn
