import yaml
from aws_cdk import (
    core,
    aws_imagebuilder as cdk_ib,
)

OUTPOST_COMPONENT_TEMPLATE = """\
name: ConfigureOutpostSpecifics
description: Configures Outpost specific things to work with on-prem
schemaVersion: 1.0

phases:
  - name: build
    steps:
      - name: DNSResolutionForwarding
        action: CreateFile
        inputs:
          - path: /etc/systemd/resolved.conf.d/justin_tv.conf
            content: |
              # configures split-dns to forward any queries for justin.tv to baremetal DNS anycast addreses
              [Resolve]
              DNS={nameservers}
              Domains={domains}
      - name: ConfigureResolvedRoutingDomains
        action: CreateFile
        inputs:
          - path: /etc/cloud/cloud.cfg.d/99_network_resolved.cfg
            content: |
              # This is the default config for cloud-init, edited to automatically add routing domains for systemd-resolved
              # https://github.com/canonical/cloud-init/blob/0d90596b56db5d306125ead08c571fc8d44d528e/cloudinit/net/netplan.py
              network:
                version: 2
                ethernets:
                  all-en:
                    match:
                      name: "en*"
                    dhcp4: true
                    nameservers:
                      # systemd-resolved syntax to configure routing domains
                      # see: https://www.freedesktop.org/software/systemd/man/resolved.conf.html#Domains=
                      search: ["~."]
                  all-eth:
                    match:
                      name: "eth*"
                    dhcp4: true
                    nameservers:
                      # systemd-resolved syntax to configure routing domains
                      # see: https://www.freedesktop.org/software/systemd/man/resolved.conf.html#Domains=
                      search: ["~."]
"""


class OutpostBuildComponentsConstruct(core.Construct):
    """Auto generated build component to install further list of packages"""

    def __init__(self, scope: core.Construct, id: str,
            nameservers=None, domains=None,
            component_version='1.0.0', **kwargs) -> None:
        super().__init__(scope, id, **kwargs)

        if nameservers is None:
            nameservers = ['10.254.0.3', '10.254.0.8']

        if domains is None:
            domains = ['justin.tv']

        name = 'Outpost'

        # creates systemd-resolved configuration syntax for routing domains
        domains = [f'~{d}' for d in domains]

        data = OUTPOST_COMPONENT_TEMPLATE.format(domains=' '.join(domains), nameservers=' '.join(nameservers))
        self._component = cdk_ib.CfnComponent(self,
                                              "BuildCmpt{}{}".format(name, component_version),
                                              name='{}-{}'.format(core.Stack.of(self).stack_name, name),
                                              description=yaml.load(data, Loader=yaml.FullLoader).get('description'),
                                              platform="Linux",
                                              version=component_version,
                                              data=data,
                                              )

    @property
    def component_arns(self):
        return [self._component.attr_arn]
