from aws_cdk import (
    core,
    aws_s3 as cdk_s3,
    aws_iam as cdk_iam,
)


class ScaffoldingConstruct(core.Construct):
    """Construct helper that builds required scaffolding resources
    like the s3 debug bucket and build infra IAM instance profile.
    exposes IAM and S3 objects to reference metadata in subsequent steps.
    """
    def __init__(self, scope: core.Construct, id: str, **kwargs) -> None:
        super().__init__(scope, id, **kwargs)

        # create S3 resources
        self._logging_bucket = cdk_s3.Bucket(self,
                                             "Logs",
                                             block_public_access=cdk_s3.BlockPublicAccess.BLOCK_ALL
                                             )

        # create IAM resources
        role = cdk_iam.Role(self, "BuildInstanceRole", assumed_by=cdk_iam.ServicePrincipal("ec2.amazonaws.com"))
        # The EC2 img that builds your image needs access to SSM, CW, S3, VPC reads, and EC2 distribution permissions.
        for p in self._managed_policies():
            role.add_managed_policy(p)

        # create inline policy
        cdk_iam.Policy(self, "Ec2S3Policy",
                       roles=[role],
                       statements=[cdk_iam.PolicyStatement(
                           actions=["s3:PutObject", "s3:GetObject"],
                           resources=[self._logging_bucket.bucket_arn, "{}/*".format(self._logging_bucket.bucket_arn)]
                       )]
                       )

        self._instance_profile = cdk_iam.CfnInstanceProfile(self,
                                                            "InstanceProfile",
                                                            instance_profile_name=core.Stack.of(self).stack_name,
                                                            roles=[role.role_name]
                                                            )

    def _managed_policies(self):
        managed_policies = [
            "AmazonSSMManagedInstanceCore",
            "EC2InstanceProfileForImageBuilder",
            "AmazonVPCReadOnlyAccess",
            "Ec2ImageBuilderCrossAccountDistributionAccess",
        ]

        return [cdk_iam.ManagedPolicy.from_aws_managed_policy_name(p) for p in managed_policies]

    @property
    def instance_profile(self):
        return self._instance_profile

    @property
    def logging_bucket(self):
        return self._logging_bucket
