import argparse
import logging
from . import cert
from . import emf
from . import ddb
from . import sandstorm

from certzero.constants import *

def sandstorm_scan():
    import json, os
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument('-v', '--verbose', action='store_true', default=False,
                        help='turn on verbose logging')

    args = parser.parse_args()

    if args.verbose:
        level = logging.DEBUG
    else:
        level = logging.INFO
    logging.basicConfig(level=level, format='%(message)s')

    dynamo = ddb.Session()
    ss = sandstorm.Session()
    logs = emf.Session(logStreamName="sandstorm_scan-{}".format(os.getpid()))

    for item in dynamo.dump():
        logging.debug("parsing secret {}".format(item.name))
        secret = ss.get_secret(item)
        certificate = cert.Certificate(secret)
        logging.debug("days to expire: {}".format(certificate.days_to_expiry()))

        dimensions = [
            [
                emf.Dimension('subject', certificate.subject()),
                emf.Dimension('role', 'sandstorm-lambda'),
            ],
        ]

        metrics = [ emf.Metric('days_to_expiry', certificate.days_to_expiry()) ]

        properties = {
                    'issuer': certificate.issuer(),
                    'not_valid_before': str(certificate.not_valid_before()),
                    'not_valid_after': str(certificate.not_valid_after()),
                }

        m = logs.create_message(metrics=metrics, dimensions=dimensions, properties=properties)
        m.send(proto='lambda')


def filesystem_scan():
    import json, os
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument('-c', '--configdir', default=CONF_DIR,
                        help='directory to read cert metadata from')
    parser.add_argument('-p', '--pop', default='sfo01',
                        help='ignored, here for compat, delete after puppet update')
    parser.add_argument('-d', '--debug', action='store_true', default=False,
                        help='debug dry run')
    parser.add_argument('-v', '--verbose', action='store_true', default=False,
                        help='turn on verbose logging')

    args = parser.parse_args()

    if args.verbose:
        level = logging.DEBUG
    else:
        level = logging.INFO
    logging.basicConfig(level=level, format='%(message)s')

    logs = emf.Session()

    if args.debug:
        proto = 'debug'
    else:
        proto = 'udp'

    certs = list()
    for f in os.listdir(args.configdir):
        config_path = os.path.join(args.configdir, f)
        try:
            with open(config_path, 'r') as config:
                c = json.load(config)
                certs.append(c['cert_path'])
        except:
            logging.exception("err reading {}".format(config_path))

    for f in certs:
        try:
            with open(f, 'r') as c:
                contents = ''.join(c.readlines())
                certificate = cert.Certificate(contents.encode())
        except:
            logging.exception("err reading {}".format(cert))
            continue

        dimensions = [
            [
                emf.Dimension('subject', certificate.subject()),
                emf.Dimension('environment', logs.twitch_environment),
                emf.Dimension('role', logs.twitch_role),
                emf.Dimension('pop', logs.pop),
                emf.Dimension('hostname', logs.hostname),
            ],
            [
                emf.Dimension('subject', certificate.subject()),
                emf.Dimension('environment', logs.twitch_environment),
                emf.Dimension('role', logs.twitch_role),
                emf.Dimension('pop', logs.pop),
            ],
        ]

        metrics = [ emf.Metric('days_to_expiry', certificate.days_to_expiry()) ]

        properties = {
                    'issuer': certificate.issuer(),
                    'not_valid_before': str(certificate.not_valid_before()),
                    'not_valid_after': str(certificate.not_valid_after()),
                }

        m = logs.create_message(metrics=metrics, dimensions=dimensions, properties=properties)
        m.send(proto=proto)


def generate_report():
    import csv, locale, sys
    parser = argparse.ArgumentParser(description=__doc__)
    parser.add_argument('-m', '--min-days', default=0, type=int,
                        help='minimum days remaining to print')
    parser.add_argument('-v', '--verbose', action='store_true', default=False,
                        help='turn on verbose logging')

    args = parser.parse_args()

    if args.verbose:
        level = logging.DEBUG
    else:
        level = logging.INFO
    logging.basicConfig(level=level, format='%(message)s')

    dynamo = ddb.Session()
    ss = sandstorm.Session()

    locale.setlocale(locale.LC_ALL, 'en_US')
    writer = csv.writer(sys.stdout)
    writer.writerow(['Subject', 'Start date', 'All Day Event'])

    for item in dynamo.dump():
        logging.debug("parsing secret {}".format(item.name))
        secret = ss.get_secret(item)
        certificate = cert.Certificate(secret)

        if certificate.days_to_expiry() > args.min_days:
            writer.writerow([item.name, certificate.not_valid_after().strftime('%x'), True])
