import boto3
import datetime
import logging
import socket
from botocore.config import Config

from certzero.constants import *


class Session(object):
    def __init__(self, role=LOGS_ROLE, logGroupName=LOG_GROUP, logStreamName=socket.gethostname()):
        self.logGroupName = logGroupName
        self.logStreamName = logStreamName

        config = Config(
            retries = {
                'mode': BOTO_RETRY_MODE,
                'max_attempts': BOTO_MAX_ATTEMPTS,
            }
        )

        if role:
            logging.debug("creating cloudwatch logs client with role {}".format(role))
            c = boto3.client('sts', config=config)
            r = c.assume_role(RoleArn=role, RoleSessionName='cwlogs')
            session = boto3.Session(
                aws_access_key_id=r['Credentials']['AccessKeyId'],
                aws_secret_access_key=r['Credentials']['SecretAccessKey'],
                aws_session_token=r['Credentials']['SessionToken']
                )
            self.client = session.client('logs', config=config)
        else:
            logging.debug("creating cloudwatch logs client with default role")
            self.client = boto3.client('logs', config=config)

        # create log group if it doesnt exist
        resp = self.client.describe_log_groups(logGroupNamePrefix=self.logGroupName)
        if self.logGroupName not in [ lg['logGroupName'] for lg in resp['logGroups'] ]:
            self.client.create_log_group(logGroupName=self.logGroupName)

        # create log stream if it doesnt exist
        resp = self.client.describe_log_streams(logGroupName=self.logGroupName, logStreamNamePrefix=self.logStreamName)
        if self.logStreamName not in [ lg['logStreamName'] for lg in resp['logStreams'] ]:
            self.client.create_log_stream(logGroupName=self.logGroupName, logStreamName=self.logStreamName)
            resp = self.client.describe_log_streams(logGroupName=self.logGroupName, logStreamNamePrefix=self.logStreamName)

        # get initial sequence token for log stream
        for ls in resp['logStreams']:
            if ls['logStreamName'] == self.logStreamName: self.sequencetoken = ls.get('uploadSequenceToken', None)


    def put_log_events(self, logs):
        logEvents = list()
        for log in logs:
            ts = int(datetime.datetime.timestamp(datetime.datetime.now(datetime.timezone.utc)) * 1000)
            logEvents.append({ 'timestamp': ts, 'message': log })

        if self.sequencetoken:
            resp = self.client.put_log_events(
                    logGroupName=self.logGroupName,
                    logStreamName=self.logStreamName,
                    logEvents=logEvents,
                    sequenceToken=self.sequencetoken)
        else:
            resp = self.client.put_log_events(
                    logGroupName=self.logGroupName,
                    logStreamName=self.logStreamName,
                    logEvents=logEvents)


        self.sequencetoken = resp['nextSequenceToken']

