import boto3
import botocore
from botocore.config import Config
from datetime import datetime
import logging

from certzero.constants import *


class Session(object):
    def __init__(self, role=METRICS_ROLE):
        config = Config(
            retries = {
                'mode': BOTO_RETRY_MODE,
                'max_attempts': BOTO_MAX_ATTEMPTS,
            }
        )
        
        if role:
            logging.debug("creating cloudwatch client with role {}".format(role))
            c = boto3.client('sts', config=config)
            r = c.assume_role(RoleArn=role, RoleSessionName='cwmetrics')
            session = boto3.Session(
                aws_access_key_id=r['Credentials']['AccessKeyId'],
                aws_secret_access_key=r['Credentials']['SecretAccessKey'],
                aws_session_token=r['Credentials']['SessionToken']
                )
            self.client = session.client('cloudwatch', config=config)
        else:
            logging.debug("creating cloudwatch client with default role")
            self.client = boto3.client('cloudwatch', config=config)

        self.queue = list()

    def enqueue(self, metric):
        self.queue.append(metric)

        if len(self.queue) > MAX_METRICS:
            self.put_metrics(self.queue)

    def put_metrics(self, metrics, namespace='CertZero'):
        try:
            self.client.put_metric_data(
                Namespace=namespace,
                MetricData=metrics
            )
        except botocore.exceptions.ClientError as error:
            if error.response['Error']['Code'] == 'RequestEntityTooLarge':
                logging.info("needed to split metric put of length {}".format(len(metrics)))
                self.instrument_err(len(metrics))
                self.put_metrics(metrics[:len(metrics // 2)])
                self.put_metrics(metrics[len(metrics // 2):])

    def instrument_err(self, bucketsize, namespace='VidCS', service='CertZero'):
        d = [ Dimension('service', service), ]
        m = Metric( MetricName='RequestEntityTooLarge',
                    Dimensions=d,
                    Value=bucketsize,
                    Unit='Count',
                  )
        self.client.put_metric_data(Namespace=namespace, MetricData=[m])


class Metric(dict):
    def __init__(self, **kwargs):
        self.update(kwargs)
        self['MetricName'] = kwargs.get('MetricName', 'days_to_expiry')
        self['Dimensions'] = kwargs.get('Dimensions', list())
        self['Value'] = kwargs.get('Value', 0)
        self['Timestamp'] = kwargs.get('Timestamp', datetime.now())
        self['Unit'] = kwargs.get('Unit', None)


class Dimension(dict):
    def __init__(self, d):
        key, value = d
        self['Name'] = key
        self['Value'] = value

