import boto3
from botocore.config import Config
from datetime import datetime
import logging
import re

from certzero.constants import *


class Session(object):
    def __init__(self, role=SANDSTORM_ROLE):
        config = Config(
            retries = {
                'mode': BOTO_RETRY_MODE,
                'max_attempts': BOTO_MAX_ATTEMPTS,
            }
        )

        logging.debug("creating ddb client with role {}".format(role))
        c = boto3.client('sts', config=config)
        r = c.assume_role(RoleArn=role, RoleSessionName='ddb')
        session = boto3.Session(
            aws_access_key_id=r['Credentials']['AccessKeyId'],
            aws_secret_access_key=r['Credentials']['SecretAccessKey'],
            aws_session_token=r['Credentials']['SessionToken']
            )
        self.client = session.client('dynamodb', config=config)

    def scan(self, table):
        logging.debug('dumping ddb contents')
        resp = self.client.scan(TableName=table)
        for i in resp['Items']: yield i
        while 'LastEvaluatedKey' in resp:
            resp = self.client.scan(TableName='sandstorm-production',
                                    ExclusiveStartKey=resp['LastEvaluatedKey'])
            for i in resp['Items']: yield i

    def dump(self, table='sandstorm-production'):
        for item in self.scan(table):
            for path in SECRET_PATHS:
                p = re.compile(path)
                if p.match(item['name']['S']):
                    yield(Item(item))


class Item(object):
    def __init__(self, item):
        def to_bin(b): return b
        def to_bool(b): return bool(b)
        def to_str(s): return str(s)
        def to_time(t): return datetime.fromtimestamp(float(t))

        types = {
            'B': to_bin,
            'BOOL': to_bool,
            'N': to_time,
            'S': to_str,
            }

        for obj in item:
            for type, data in item[obj].items():
                func = types[type]
                self.__setattr__(obj, func(data))

