import datetime
import json
import logging
import socket
import subprocess

from certzero.constants import *

CMD = ['facter', '--json', '-p', '--external-dir=/etc/facter/facts.d' ]
FACTS = [ 'pop',
          'twitch_role',
          'twitch_environment',
          'hostname'
        ]


class Session(object):
    def __init__(self, **kwargs):
        p = subprocess.Popen(CMD + FACTS, stdout=subprocess.PIPE)
        p.wait()
        out = b''.join(p.stdout.readlines())
        facts = json.loads(out.decode())

        self.pop = facts.get('pop', None) 
        self.twitch_role = facts.get('twitch_role', None) 
        self.twitch_environment = facts.get('twitch_environment', None) 
        self.hostname = facts.get('hostname', None) 

        self.namespace = kwargs.get('namespace', 'CertZero')
        self.logGroupName = kwargs.get('logGroupName', LOG_GROUP)
        self.logStreamName = kwargs.get('logStreamName', '.'.join([self.hostname, self.pop]))

    def create_message(self, **kwargs):
        metrics = kwargs.get('metrics')
        dimensions = kwargs.get('dimensions', [])
        ctx_properties = kwargs.get('properties', {})
        ctx_metrics = list()
        ctx_dimensions = list()

        for metric in metrics:
            m = metric.copy()
            ctx_properties.update({ m['Name']: m.pop('Value') })
            ctx_metrics.append(m)

        for rollup in dimensions:
            # [ [foo], [foo, bar] ]
            for d in rollup:
                ctx_properties.update({ d['Name']: d['Value'] })
            ctx_dimensions.append([ i['Name'] for i in rollup ])

        return MetricContext(
            logGroupName=self.logGroupName,
            logStreamName=self.logStreamName,
            namespace=self.namespace,
            dimensions=ctx_dimensions,
            metrics=ctx_metrics,
            properties=ctx_properties,
            )

class Metric(dict):
    def __init__(self, name, value, unit=None):
        self['Name'] = name
        self['Value'] = value
        if unit: self['Unit'] = unit

class Dimension(dict):
    def __init__(self, name, value):
        self['Name'] = name
        self['Value'] = value

class MetricContext(dict):
    def __init__(self, **kwargs):
        properties = kwargs.get('properties', {})
        ts = int(datetime.datetime.timestamp(datetime.datetime.now(datetime.timezone.utc)) * 1000)
        self.update(
            {
              '_aws': {
                'Timestamp': ts,
                'LogGroupName': kwargs.get('logGroupName'),
                'LogStreamName': kwargs.get('logStreamName'),
                'CloudWatchMetrics': [
                  {
                    'Namespace': kwargs.get('namespace'),
                    'Dimensions': kwargs.get('dimensions', []),
                    'Metrics': kwargs.get('metrics', []),
                  }
                ]
              },
            'certzero_version': VERSION,
            }
        )
        self.update(properties)

    def send(self, proto=EMF_PROTO, host=EMF_HOST, port=EMF_PORT):
        if proto == 'udp':
            message = json.dumps(self) + "\n"
            sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            sock.sendto(message.encode('UTF-8'), (host, port))
            sock.close()
        else:
            print(json.dumps(self, indent=2))
