#!/usr/bin/env python3
import os
import yaml
import json

from aws_cdk import core

from cloudwatch_alarm_cdk.cloudwatch_alarm_cdk_stack import (
    CloudwatchAlarmCdkStack
)

from cloudwatch_alarm_cdk.common import (
    Common
)

def region_config():
    """
    Return the tfstate json object for the region mapping
    """
    with open('terraform.tfstate') as f:
        return json.load(f)

def read_config(account):
    """
    Return a list of dictionaries.

    Each item is a dict representation of the config
    """
    # get dashboards that want to be templated
    with open('alarms/{}/config.yaml'.format(account)) as f:
        dashboard_metadata = yaml.load(f, Loader=yaml.Loader)
    return dashboard_metadata


if __name__ == "__main__":
    app = core.App()
    account = os.environ['AWS_ACCOUNT_NAME']
    account_num = os.environ['AWS_ACCOUNT_NUM']
    # get region mapping data from tf source of truth
    region_mapping_data = region_config()
    cw_regions = region_mapping_data.get('outputs').get('in_commission_datacenters').get('value').keys()
    # Shared prepend value to detect republisher rules
    rule_prepend = "CWAlarmsCDK"
    # create common resources (i.e: metric math republisher scaffolding)
    common_dict = {}
    for r in cw_regions:
        common_dict[r] = Common(app, f"CWAlarmsCDKCommon-{r}", rule_prepend, env={'account': account_num, 'region': r})
    alarm_configs = read_config(account)
    for alarm_name, alarm_cfg in alarm_configs.items():
        # The regions to replicate by is either
        # a hardcoded list under IncludeRegions
        if 'IncludeRegions' in alarm_cfg:
            regions = alarm_cfg.get('IncludeRegions')
        # Or if unspecified, the default_regions - ExcludeRegions
        else:
            regions = ([item for item in cw_regions
                       if item not in alarm_cfg.get('ExcludeRegions')]
                       if 'ExcludeRegions' in alarm_cfg else cw_regions)
        alarm_cfg['AlarmName'] = alarm_name
        alarm_cfg['AccountName'] = account
        for r in regions:
            CloudwatchAlarmCdkStack(app,
                                    "CWAlarmsCDK-{}-{}".format(alarm_name, r),
                                    alarm_cfg,
                                    region_mapping=region_mapping_data.get('outputs').get('in_commission_datacenters').get('value'),
                                    common={'republisher_lambda': common_dict[r].get_lambda_function(), 'rule_prepend': rule_prepend},
                                    env={'account': account_num, 'region': r},
                                    termination_protection=True
                                    )

    app.synth()
