from aws_cdk import (
    core,
    aws_cloudwatch as cdk_cw,
    aws_events as cdk_events,
    aws_events_targets as cdk_targets,
    aws_logs as cdk_logs,
    aws_sns as cdk_sns,
    aws_sns_subscriptions as cdk_sub
)

import requests
import logging
import sys
import yaml
import json

CONSUL_API = "https://api.us-west-2.prod.consul.live-video.a2z.com"

logger = logging.getLogger()
logging.basicConfig(stream=sys.stdout, level=logging.INFO)


class CloudwatchAlarmCdkStack(core.Stack):
    # These are the sins we live with forever
    stupid_exceptions = {
        'eu-west-1': lambda initial_list: initial_list.append('lhr05'),  # system metrics in lhr05 are mapped to eu-west-1
        'eu-west-2': lambda initial_list: initial_list.remove('lhr05'),  # system metrics in lhr05 are not in eu-west-2
    }

    def __init__(self, scope: core.Construct, id: str,
                 alarm_cfg, region_mapping={}, common={}, **kwargs) -> None:
        super().__init__(scope, id, **kwargs)
        # Statically configured instance vars
        self.aws_region = kwargs.get('env').get('region')
        self.account_name = alarm_cfg.get('AccountName')
        self.alarm_name = alarm_cfg.get('AlarmName')
        self.tmpl_pop = alarm_cfg.get('TemplatePop', '')
        self.include_pops = alarm_cfg.get('IncludePops', [])
        self.exclude_pops = alarm_cfg.get('ExcludePops', [])
        self.region_mapping = region_mapping
        # All DCs to create alarms for in the region
        self.candidate_dcs = self.form_consul_dc_list(self.tmpl_pop)

        # Build any configured log filters before alarms (optional)
        self.build_log_filters(alarm_cfg.get('MetricFilters', {}))
        # Build any contributor insights (optional)
        self.build_contributor_insights(alarm_cfg.get('InsightRules', {}))
        # Build any metrics republishing notifications (optional)
        self.build_republisher_events(alarm_cfg.get('Republisher', {}), common)

        # Resources built for each alarm stack
        # Build SNS topics if configured (optional, but why wouldn't you)
        self.topic_mapping = self.build_topics(alarm_cfg.get('SNSActions', {}))
        # Build and replicate alarms
        self.build_alarms(alarm_cfg)

    # build_alarms: the core logic of taking the desired yaml input and creating the desired alarm
    # replicated to all pops mapped to the region.
    def build_alarms(self, alarm_cfg):
        tmpl_alarm = alarm_cfg.get('AlarmConfig')  # Assume MetricAlarm
        for dc in self.candidate_dcs:
            # If configured, only configure DCs that appear in the explicit allowlist and exclude denylist pops.
            if (len(self.include_pops) == 0 and dc not in self.exclude_pops) or dc in self.include_pops:
                # alarm name and actions dict have to be formed regardless
                new_alarm_name = f"{self.alarm_name}-{dc}"
                # Depending on kind of CW alarm, the term that's swapped out changes
                alarm_desc = tmpl_alarm.get('alarm_description').replace('TemplateRegion', dc) if not self.tmpl_pop else tmpl_alarm.get('alarm_description', '').replace('TemplatePop', dc)
                # Either the metrics is based off of an array (which includes Math Expressions)
                if alarm_cfg.get('UseMetrics', False):
                    with open(f'base-metrics/{self.account_name}/{self.alarm_name}.json') as f:
                        import_metrics = yaml.load(f, Loader=yaml.Loader)
                    # form metrics object
                    metrics_list = self._form_metric_list(import_metrics, self.tmpl_pop, dc)
                    # form actions dictionary
                    alarm = cdk_cw.CfnAlarm(self, new_alarm_name,
                                            actions_enabled=tmpl_alarm.get('actions_enabled', True),
                                            alarm_actions=self.topic_mapping.get('Alarm', []),
                                            alarm_description=alarm_desc,
                                            alarm_name=new_alarm_name,
                                            metrics=metrics_list,
                                            comparison_operator=tmpl_alarm.get('comparison_operator', 'GreaterThanThreshold'),
                                            threshold=tmpl_alarm.get('threshold'),  # not setting a default on purpose so cdk will error if yaml not configured
                                            datapoints_to_alarm=tmpl_alarm.get('datapoints_to_alarm', 1),
                                            evaluation_periods=tmpl_alarm.get('evaluation_periods', 1),
                                            insufficient_data_actions=self.topic_mapping.get('InsufficientData', []),
                                            ok_actions=self.topic_mapping.get('Ok', []),
                                            treat_missing_data=tmpl_alarm.get('treat_missing_data')
                                            )
                # Or it's not a math expression
                else:
                    dms = self._form_dimensions(tmpl_alarm.get('dimensions'), dc)
                    alarm = cdk_cw.CfnAlarm(self, new_alarm_name,
                                            actions_enabled=tmpl_alarm.get('actions_enabled', True),
                                            alarm_actions=self.topic_mapping.get('Alarm', []),
                                            alarm_description=alarm_desc,
                                            alarm_name=new_alarm_name,
                                            namespace=tmpl_alarm.get('namespace'),
                                            metric_name=tmpl_alarm.get('metric_name'),
                                            dimensions=dms,
                                            comparison_operator=tmpl_alarm.get('comparison_operator', 'GreaterThanThreshold'),
                                            threshold=tmpl_alarm.get('threshold'),  # not setting a default on purpose so cdk will error if yaml not configured
                                            datapoints_to_alarm=tmpl_alarm.get('datapoints_to_alarm', 1),
                                            period=tmpl_alarm.get('period', 60),  # TODO: sane default? Or let fail
                                            statistic=tmpl_alarm.get('statistic', ''),
                                            evaluation_periods=tmpl_alarm.get('evaluation_periods', 1),
                                            insufficient_data_actions=self.topic_mapping.get('InsufficientData', []),
                                            ok_actions=self.topic_mapping.get('Ok', []),
                                            treat_missing_data=tmpl_alarm.get('treat_missing_data')
                                            )
                # support anomaly detection if configured
                if tmpl_alarm.get('threshold_metric_id', None) is not None:
                    alarm.add_property_deletion_override('Threshold')
                    alarm.add_property_override('ThresholdMetricId', tmpl_alarm.get('threshold_metric_id'))

    # _form_dimensions is a helper to create DimensionProperty objects out of metrics that have dimensions.
    # This is required to form a proper CfnAlarm object
    def _form_dimensions(self, dimension_list, dc):
        ret = []
        if dimension_list is not None:
            for d in dimension_list:
                value = dc if d['Name'] == 'pop' else d['Value']
                ret.append(cdk_cw.CfnAlarm.DimensionProperty(name=d['Name'], value=value))
        return ret

    # _form_metric is a helper to create a MetricStatProperty object.
    # This is required to form a proper CfnAlarm object
    def _form_metric(self, mstat, dc):
        metric = cdk_cw.CfnAlarm.MetricProperty(namespace=mstat.get('Metric').get('Namespace'),
                                                metric_name=mstat.get('Metric').get('MetricName'),
                                                dimensions=self._form_dimensions(mstat.get('Metric').get('Dimensions', []), dc)
                                                )
        return cdk_cw.CfnAlarm.MetricStatProperty(metric=metric,
                                                  period=mstat.get('Period'),
                                                  stat=mstat.get('Stat'),
                                                  unit=mstat.get('Unit', 'None')
                                                  )

    # _form_metric_list is a helper to create MetricDataQueryProperty objects out of a metrics array.
    # This is required to form a proper CfnAlarm object
    def _form_metric_list(self, metrics_list, tmpl_pop, dc):
        ret = []
        for m in metrics_list:
            if 'Label' in m:
                label = m.get('Label') if not tmpl_pop else m.get('Label').replace(tmpl_pop, dc)
            # the formed MetricDataQueryProperty differs depending on if the metric is a math expression of a MetricStat
            if 'Expression' in m:
                # search expressions can include tmpl_pop
                expr = m.get('Expression') if not tmpl_pop else m.get('Expression').replace(tmpl_pop, dc)
                ret.append(cdk_cw.CfnAlarm.MetricDataQueryProperty(id=m.get('Id'),
                                                                   label=label,
                                                                   period=m.get('Period', 60),
                                                                   expression=expr,
                                                                   return_data=m.get('ReturnData')))
            else:
                metric = self._form_metric(m.get('MetricStat'), dc)
                ret.append(cdk_cw.CfnAlarm.MetricDataQueryProperty(id=m.get('Id'),
                                                                   label=label,
                                                                   metric_stat=metric,
                                                                   return_data=m.get('ReturnData')))
        return ret

    # build_contributor_insigths: an optional plugin that precreates a contributor insight rule that
    # a following alarm config may use to create replicated alarms
    def build_contributor_insights(self, insight_configs):
        required_keys = ['rule_state', 'rule_body']
        for fName, iConfig in insight_configs.items():
            if all(key in iConfig for key in required_keys):
                insightID = f"{fName}-{self.aws_region}"  # ID is just the region appended to the metric name
                cdk_cw.CfnInsightRule(self, insightID,
                                      rule_name=fName,
                                      rule_state=iConfig.get('rule_state'),
                                      rule_body=json.dumps(iConfig.get('rule_body'))
                                      )
            else:
                logger.warning(f"Will not create contributor insight rule for {fName}. All required keys ({required_keys}) necessary")

    # build_log_filters: an optional plugin that precreates a cloudwatch metric log filter that
    # a following alarm config may use to create replicated alarms.
    def build_log_filters(self, filter_configs):
        required_keys = ['log_group', 'filter_pattern', 'metric_namespace', 'default_value']
        for fName, fConfig in filter_configs.items():
            if all(key in fConfig for key in required_keys):
                filterID = f"{fName}-{self.aws_region}"  # ID is just the region appended to the metric name
                cdk_logs.CfnMetricFilter(self, filterID,
                                         log_group_name=fConfig.get('log_group'),
                                         filter_pattern=fConfig.get('filter_pattern'),
                                         metric_transformations=[
                                            cdk_logs.CfnMetricFilter.MetricTransformationProperty(
                                                metric_value=fConfig.get('metric_value'),
                                                metric_namespace=fConfig.get('metric_namespace'),
                                                metric_name=fName,
                                                default_value=fConfig.get('default_value', 0)
                                            )
                                         ]
                                         )
            else:
                logger.warning(f"Will not create log filter for {fName}. All required keys ({required_keys}) necessary")

    # build_republisher_events: an optional plugin that precreates an EventBridge event that feeds into the region's
    # republisher lambda to create a republished metric that a following alarm config may use to create replicated alarms.
    def build_republisher_events(self, republisher_config, republisher_metadata):
        # create a new bus to plop rules so we don't hit limits after a handful of alarms
        if republisher_config:
            for dc in self.candidate_dcs:
                # If configured, only configure DCs that appear in the explicit allowlist and exclude denylist pops.
                if (len(self.include_pops) == 0 and dc not in self.exclude_pops) or dc in self.include_pops:
                    rule = cdk_events.Rule(self, f"{self.alarm_name}RuleTarget{dc}", schedule=cdk_events.Schedule.expression('cron(0/1 * * * ? *)'))
                    target = self._form_event_rule_target(dc, republisher_config, republisher_metadata)
                    if target is not None:
                        rule.add_target(target)

    # _form_event_rule_target builds the lambda function target to be added to the event rule.
    def _form_event_rule_target(self, dc, republisher_config, republisher_metadata):
        required_keys = ['expression', 'label']
        if not all(key in republisher_config for key in required_keys):
            logger.warning(f"Will not create republisher config for {self.alarm_name}. All required keys ({required_keys}) necessary")
            return None
        base_dimensions = [{"Name": "pop", "Value": dc}]
        rule_target_input = {
            "delayMinutes": 4,
            "period": 60,
            "queries": [
                {
                    "gmdParams": {
                        "MetricDataQueries": [
                            {
                                "Id": "e1",
                                "Expression": republisher_config.get('expression').replace(self.tmpl_pop, dc),
                                "Label": republisher_config.get('label')
                            }
                        ]
                    },
                    "metricTemplate": {
                        "Namespace": "LambdaAggregates",  # hardcoded to centralize metrics destination.
                        "Dimensions": list(filter(None, base_dimensions + republisher_config.get('dimensions', {}))),
                        "MetricName": republisher_config.get('metric_name', '$label')
                    }
                }
            ]
        }
        return cdk_targets.LambdaFunction(republisher_metadata.get('republisher_lambda'),
                                          event=cdk_events.RuleTargetInput.from_object(rule_target_input)
                                          )

    # build_topics: return mapping of topics to list of actions to associate it with
    # (e.g: {'Ok': ['<tpc1>', '<tpc2>'], 'InsufficientData': [], 'Alarm': ['<tpc2>']})
    def build_topics(self, sns_configs):
        # Decide whether SNS topics need to be created
        ret = {'Ok': [], 'InsufficientData': [], 'Alarm': []}
        if any(sns_configs):
            # loop through each
            for tname, tcfg in sns_configs.items():
                name = f"{tname}-{self.aws_region}"
                new_topic = cdk_sns.Topic(self, name, topic_name=name)
                if 'Endpoints' in tcfg:
                    for e in tcfg.get('Endpoints'):
                        new_topic.add_subscription(cdk_sub.UrlSubscription(e))
                for action in tcfg.get('Actions'):
                    ret[action].append(new_topic.topic_arn)
        return ret

    # form_consul_dc_list is a public helper that filters the full list of consul DCs
    # with the list of DCs for a particular region and applies any esoteric exceptions we ahve
    def form_consul_dc_list(self, tmpl_pop):
        if not tmpl_pop:
            return [self.aws_region]
        all_consul_dcs = self._consul_dcs()
        dc_list = list(set(all_consul_dcs) & set(self.region_mapping.get(self.aws_region)))
        dc_list.append(self.aws_region)
        # perform the exception lambda if necessary otherwise just return the normal DC list
        self.stupid_exceptions.get(self.aws_region)(dc_list) if self.aws_region in self.stupid_exceptions.keys() else dc_list
        return dc_list

    # _consul_dcs is a helper to dynamically discover the list of DCs we have registered in consul.
    def _consul_dcs(self):
        try:
            resp = requests.get(f"{CONSUL_API}/v1/catalog/datacenters")
            return resp.json()
        except Exception as e:
            logger.warning(f"Failed to query consul, automatic pop discovery will not work: {e}")
            return {}
