from aws_cdk import (
    core,
    aws_iam as cdk_iam,
    aws_lambda as cdk_lambda
)
import logging
import os
import sys

logger = logging.getLogger()
logging.basicConfig(stream=sys.stdout, level=logging.INFO)


# Common is a stack of resources that are only meant to be instantiated
# once per region or overall and not meant to be repeated per alarm stack
class Common(core.Stack):
    republisher_dir = "republisher"

    def __init__(self, scope: core.Construct, id: str,
                 rule_prepend, **kwargs) -> None:
        super().__init__(scope, id, **kwargs)
        self.aws_region = kwargs.get('env').get('region')
        self.account_num = kwargs.get('env').get('account')
        self.rule_prepend = rule_prepend
        self.iam_role = self.configure_iam()
        self.lambda_function = self.configure_lambda()

    def get_lambda_function(self):
        return self.lambda_function

    # Configure a lambda IAM role per region to allow
    # each lambda to publish to cloudwatch
    def configure_iam(self):
        policy_doc = cdk_iam.PolicyDocument()
        # republisher needs logging permission to log emf logs
        policy_doc.add_statements(cdk_iam.PolicyStatement(
                                  actions=["logs:CreateLogGroup", "logs:CreateLogStream", "logs:PutLogEvents"],
                                  resources=[f"arn:aws:logs:{self.aws_region}:{self.account_num}:log-group:/aws/lambda/CWAlarmsCDK*"]
                                  ))
        # republisher needs to GetMetricData
        policy_doc.add_statements(cdk_iam.PolicyStatement(
            actions=["cloudwatch:GetMetricData"],
            resources=["*"]
        ))
        policy = cdk_iam.ManagedPolicy(self, f"republisherPolicy{self.aws_region}", document=policy_doc)
        return cdk_iam.Role(self, f"republisherRole{self.aws_region}",
                            assumed_by=cdk_iam.ServicePrincipal("lambda.amazonaws.com"),
                            managed_policies=[policy])

    # Configures lambda per region. code is referenced as s3 asset to local path.
    # Adds permission to allow cdk configured eventbridges
    def configure_lambda(self):
        lambda_function = cdk_lambda.Function(self, f"lambdaFxn{self.aws_region}",
                                              code=cdk_lambda.Code.from_asset(f"{os.getcwd()}/{self.republisher_dir}"),
                                              handler="cloudWatchMetricMathRepublisher.handler",
                                              role=self.iam_role,
                                              runtime=cdk_lambda.Runtime("nodejs12.x"),
                                              timeout=core.Duration.seconds(60),
                                              )
        # republisher needs to allow any cdk configured eventbridge event following a particular substring
        lambda_function.add_permission(f"lambdaFxnEventPermission{self.aws_region}",
                                       principal=cdk_iam.ServicePrincipal("events.amazonaws.com"),
                                       action="lambda:InvokeFunction",
                                       source_arn=f"arn:aws:events:{self.aws_region}:{self.account_num}:rule/{self.rule_prepend}*"
                                       )
        return lambda_function
